#
# Just Triangle Faces (.jtf) exporter for Blender
#

import bpy
import bmesh
import struct
import array
from bpy_extras.io_utils import ExportHelper
from bpy.types import Operator
from bpy.props import StringProperty

bl_info = {
    "name": "JTF format",
    "author": "Kostas Michalopoulos (badsector)",
    "version": (0, 1, 1),
    "blender": (2, 80, 0),
    "location": "File > Import-Export > Just Triangle Faces",
    "description": "Import-Export JTF files",
    "warning": "",
    "description": "Export JTF mesh files",
    "support": "COMMUNITY",
    "category": "Import-Export",
}

def export_jtf_file(context, path):
    scene = context.scene
    depsgraph = context.evaluated_depsgraph_get()
    data = []
    vertexCount = 0
    for obj in context.selected_objects:
        if obj.type != 'MESH':
            continue
        evob = obj.evaluated_get(depsgraph)
        mesh = None
        try:
            mesh = evob.to_mesh()
        except RuntimeError:
            continue
        mesh.calc_loop_triangles()
        uv_layer = mesh.uv_layers.active.data
        for tri in mesh.loop_triangles:
            for idx in tri.loops:
                uv = uv_layer[idx].uv
                v = mesh.vertices[mesh.loops[idx].vertex_index]
                data.append(v.co[0])
                data.append(v.co[2])
                data.append(-v.co[1])
                data.append(v.normal[0])
                data.append(v.normal[2])
                data.append(-v.normal[1])
                data.append(uv[0])
                data.append(1-uv[1])
                vertexCount = vertexCount + 1
        evob.to_mesh_clear()
    file = open(path, "wb")
    head = struct.pack('<BBBBii', 74, 84, 70, 33, 0, int(vertexCount/3))
    file.write(head)
    floats = array.array('f', data)
    floats.tofile(file)
    file.close()

class ExportJTF(Operator, ExportHelper):
    """Save triangle mesh data with UV and normals"""
    bl_space_type = 'FILE_BROWSER'
    bl_region_type = 'TOOL_PROPS'
    bl_label = "Geometry"
    bl_parent_id = "FILE_PT_operator"
    bl_idname = "export_mesh.jtf"
    bl_label = "Export JTF"

    filename_ext = ".jtf"

    filter_glob: StringProperty(
        default="*.jtf",
        options={'HIDDEN'},
        maxlen=255,
    )

    def execute(self, context):
        export_jtf_file(context, self.filepath)
        return {'FINISHED'}

def menu_func_export(self, context):
    self.layout.operator(ExportJTF.bl_idname, text="Just Triangle Faces (.jtf)")

def register():
    bpy.utils.register_class(ExportJTF)
    bpy.types.TOPBAR_MT_file_export.append(menu_func_export)

def unregister():
    bpy.utils.unregister_class(ExportJTF)
    bpy.types.TOPBAR_MT_file_export.remove(menu_func_export)

if __name__ == "__main__":
    register()
