#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

typedef enum
{
    NONE,
    FLAT,
    THIN_SUNKEN,
    THIN_RAISED,
    THICK_SUNKEN,
    THICK_RAISED,
    ETCHED,
    EDITOR
} style_t;

typedef struct _state_t
{
    char* caption;
    style_t style;
} state_t;

static void update_prefsize(ff_window_t box)
{
    state_t* state = ff_get(box, "-state");
    int tw = 1, th = 1;
    if (state->caption[0]) {
        ff_font_text_size(ff_font_of(box), state->caption, &tw, &th);
        if (th < 15) th = 15;
        tw += 32;
        th += 8;
    }
    ff_seti(box, "container-width", tw);
    ff_seti(box, "container-height", th);
}

static void update_padding(ff_window_t box)
{
    state_t* state = ff_get(box, "-state");
    int tw, th, ntb, tb;
    if (state->caption[0]) {
        ff_font_text_size(ff_font_of(box), "A", &tw, &th);
        tb = th < 15 ? 15 : th;
        ntb = 8;
    } else ntb = tb = 0;
    tw = 0;
    switch (state->style) {
    case NONE: ntb = 0; break;
    case THIN_RAISED:
    case THIN_SUNKEN:
    case FLAT: tw = 1;  break;
    case THICK_RAISED:
    case THICK_SUNKEN:
    case ETCHED:
    case EDITOR: tw = 2; break;
    default: tw = 2;
    }
    ff_padding(box, ntb + tw, (tb ? tb : ntb) + tw, ntb + tw, ntb + tw);
}

static int set_handler(void* box, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(box, "-state");
    if (!strcmp(nv->name, "style")) {
        const char* style = nv->value;
        if (!style) style = "etched";
        if (!strcmp(style, "none"))
            state->style = NONE;
        else if (!strcmp(style, "flat"))
            state->style = FLAT;
        else if (!strcmp(style, "thin-raised"))
            state->style = THIN_RAISED;
        else if (!strcmp(style, "thin-sunken"))
            state->style = THIN_SUNKEN;
        else if (!strcmp(style, "thick-raised"))
            state->style = THICK_RAISED;
        else if (!strcmp(style, "thick-sunken"))
            state->style = THICK_SUNKEN;
        else if (!strcmp(style, "editor"))
            state->style = EDITOR;
        else
            state->style = ETCHED;
        update_padding(box);
        ff_paint(box);
        return FF_YES;
    }
    if (!strcmp(nv->name, "caption")) {
        free(state->caption);
        state->caption = ff_strdup(nv->value);
        update_padding(box);
        update_prefsize(box);
        ff_paint(box);
        return FF_YES;
    }
    return FF_NO;
}

static int get_handler(void* box, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(box, "-state");
    if (!strcmp(nv->name, "style")) {
        switch (state->style) {
        case NONE: nv->value = "none"; break;
        case FLAT: nv->value = "flat"; break;
        case THIN_RAISED: nv->value = "thin-raised"; break;
        case THIN_SUNKEN: nv->value = "thin-sunken"; break;
        case THICK_RAISED: nv->value = "thick-raised"; break;
        case THICK_SUNKEN: nv->value = "thick-sunken"; break;
        default: nv->value = "etched";
        }
        return FF_YES;
    }

    if (!strcmp(nv->name, "caption")) {
        nv->value = state->caption;
        return FF_YES;
    }
    return FF_NO;
}

static int destroy_handler(void* box, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(box, "-state");
    free(state->caption);
    free(state);
    return FF_NO;
}

static int newfont_handler(void* box, ff_event_t* ev, void* data)
{
    update_padding(box);
    update_prefsize(box);
    return 0;
}

static int paint_handler(void* box, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th, ftop;
    state_t* state = ff_get(box, "-state");
    ff_area(box, NULL, NULL, &w, &h);
    ff_color_attr(gc, box, "background", FF_3DFACE_COLOR);
    if (state->caption[0]) {
        ff_text_size(gc, state->caption, &tw, &th);
        ftop = ff_round_int(((float)th)/2.0f) - 1;
        if (ftop < 0) ftop = 0;
    } else
        tw = th = ftop = 0;
    ff_fill(gc, 0, 0, w, h);
    switch (state->style) {
    case NONE:
        break;
    case FLAT:
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_rect(gc, 0, ftop, w - 1, h - 1);
        break;
    case THIN_SUNKEN:
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 0, ftop, w - 1, ftop);
        ff_line(gc, 0, ftop, 0, h - 1);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 1, h - 1, w - 1, h - 1);
        ff_line(gc, w - 1, ftop + 1, w - 1, h - 1);
        break;
    case THIN_RAISED:
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 0, ftop, w - 1, ftop);
        ff_line(gc, 0, ftop, 0, h - 1);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 1, h - 1, w - 1, h - 1);
        ff_line(gc, w - 1, ftop + 1, w - 1, h - 1);
        break;
    case THICK_SUNKEN:
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 0, ftop, w - 2, ftop);
        ff_line(gc, 0, ftop, 0, h - 2);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 1, ftop + 1, w - 3, ftop + 1);
        ff_line(gc, 1, ftop + 1, 1, h - 3);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, w - 1, ftop, w - 1, h - 1);
        ff_line(gc, 0, h - 1, w - 1, h - 1);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_line(gc, w - 2, ftop + 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
        break;
    case THICK_RAISED:
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 0, ftop, w - 2, ftop);
        ff_line(gc, 0, ftop, 0, h - 2);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, w - 1, ftop, w - 1, h - 1);
        ff_line(gc, 0, h - 1, w - 1, h - 1);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, w - 2, ftop + 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
        break;
    case ETCHED:
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_rect(gc, 0, ftop, w - 2, h - 2);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_rect(gc, 1, ftop + 1, w - 1, h - 1);
        break;
    case EDITOR:
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 0, ftop, w - 2, ftop);
        ff_line(gc, 0, ftop, 0, h - 2);
        ff_line(gc, 1, ftop + 1, w - 3, ftop + 1);
        ff_line(gc, 1, ftop + 1, 1, h - 3);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, w - 1, ftop, w - 1, h - 1);
        ff_line(gc, 0, h - 1, w - 1, h - 1);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_line(gc, w - 2, ftop + 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
        break;
    }
    if (state->caption[0]) {
        ff_color_attr(gc, box, "background", FF_3DFACE_COLOR);
        ff_fill(gc, (w - tw) / 2 - 5, 0, (w - tw) / 2 + tw + 5, th);
        ff_color_attr(gc, box, "foreground", FF_3DTEXT_COLOR);
        ff_text(gc, (w - tw) / 2, th, state->caption);
    }
    return FF_YES;
}

ff_window_t ff_box(ff_window_t parent, const char* caption, const char* style)
{
    ff_window_t box = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    ff_set(box, "-state", calloc(1, sizeof(state_t)));
    ff_set(box, "class", (void*)"box");
    ff_link(box, FF_SET, set_handler, NULL);
    ff_link(box, FF_GET, get_handler, NULL);
    ff_set(box, "caption", (void*)caption);
    ff_set(box, "style", style ? (void*)style : "etched");
    ff_link(box, FF_DESTROY, destroy_handler, NULL);
    ff_link(box, FF_NEWFONT, newfont_handler, NULL);
    ff_link(box, FF_PAINT, paint_handler, NULL);
    update_prefsize(box);
    return box;
}
