#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

#define DOWN 0x01
#define SELECTED 0x02
#define DEFAULT 0x04
#define PUSH 0
#define TOGGLE 1
#define SELECT 2

static char* ok0glyph[] = {
    "16 10 4 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "          ....- ",
    "    ..    .   @ ",
    "   .--    .   @ ",
    "  .- -.....   @ ",
    " .-           @ ",
    ".-            @ ",
    " .            @ ",
    "  .  @@@@@@@@@@ ",
    "   . @          ",
    "    .@          "};
static char* ok1glyph[] = {
    "16 10 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "          ....- ",
    "    ..    .   * ",
    "   .--    .   * ",
    "  .- -.....   * ",
    " .-           * ",
    ".-            * ",
    " .            * ",
    "  .  ********** ",
    "   . *          ",
    "    .*          "};

static ff_bitmap_t ok0bmp, ok1bmp;

typedef struct _state_t
{
    char* caption;
    int state;
    int type;
    int group;
    ff_bitmap_t bmp;
} state_t;

static void free_bitmaps(void)
{
    if (!ok0bmp) return;
    ff_bitmap_free(ok0bmp);
    ff_bitmap_free(ok1bmp);
}

static void make_bitmaps(void)
{
    if (ok0bmp) return;
    ok0bmp = ff_xpm(ok0glyph, NULL, NULL);
    ok1bmp = ff_xpm(ok1glyph, NULL, NULL);
    ff_atexit(free_bitmaps);
}

static void update_prefsize(ff_window_t btn)
{
    state_t* state = ff_get(btn, "-state");
    int tw = 0, th = 0, bw, bh;
    if (state->caption)
        ff_font_text_size(ff_font_of(btn), state->caption, &tw, &th);
    if (state->bmp) {
        ff_bitmap_size(state->bmp, &bw, &bh);
        bw++;
        tw = tw < bw ? bw : tw;
        th += bh + (state->caption ? 4 : 1);
    }
    if (state->state & DEFAULT) {
        tw += 16;
        if (th < 12) th = 12;
    }
    ff_prefsize(btn, tw + 14, th + 6);
}

static void force_unique_selected(ff_window_t btn)
{
    ff_window_t* others = ff_children(ff_parent(btn));
    int i, group = ff_geti(btn, "group");
    for (i=0; others[i]; i++) {
        if (others[i] == btn) continue;
        if (ff_geti(others[i], "group") == group)
            ff_set(others[i], "selected", 0);
    }
    ff_free_children(others);
}

static void force_group_selection(ff_window_t btn)
{
    ff_window_t* others = ff_children(ff_parent(btn));
    int i, group = ff_geti(btn, "group"), anyfound = 0;
    for (i=0; others[i]; i++) {
        if (others[i] == btn) continue;
        if (ff_geti(others[i], "group") == group &&
            ff_get(others[i], "selected")) {
            anyfound = 1;
            break;
        }
    }
    ff_free_children(others);
    if (!anyfound) ff_seti(btn, "selected", 1);
}

static int set_handler(void* btn, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(btn, "-state");
    if (!strcmp(nv->name, "caption")) {
        free(state->caption);
        state->caption = nv->value && ((const char*)nv->value)[0] ? ff_strdup(nv->value) : NULL;
        update_prefsize(btn);
        ff_paint(btn);
        return 1;
    } else if (!strcmp(nv->name, "bitmap")) {
        ff_bitmap_t bmp = nv->value;
        ff_incref(FF_REFOBJ(bmp));
        ff_decref(FF_REFOBJ(state->bmp));
        state->bmp = bmp;
        update_prefsize(btn);
        ff_paint(btn);
        return 1;
    } else if (!strcmp(nv->name, "type")) {
        if (!strcmp(nv->value, "push"))
            state->type = PUSH;
        else if (!strcmp(nv->value, "toggle"))
            state->type = TOGGLE;
        else if (!strcmp(nv->value, "select")) {
            state->type = SELECT;
            if (!state->group) ff_seti(btn, "group", 1);
        } else return 0;
        ff_paint(btn);
        return 1;
    } else if ((!strcmp(nv->name, "selected") && state->type == SELECT) ||
               (!strcmp(nv->name, "checked") && state->type == TOGGLE)) {
        if (nv->value && !(state->state & SELECTED)) {
            state->state |= SELECTED;
            if (state->type == SELECT) force_unique_selected(btn);
            ff_paint(btn);
            ff_post(btn, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
        } else if (!nv->value && (state->state & SELECTED)) {
            state->state &= ~SELECTED;
            ff_paint(btn);
            ff_post(btn, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
            if (state->type == SELECT) force_group_selection(btn);
        }
        return 1;
    } else if (state->type == SELECT && !strcmp(nv->name, "group")) {
        state->group = (int)(ff_intptr_t)nv->value;
        force_group_selection(btn);
        return 1;
    } else if (!strcmp(nv->name, "default")) {
        if (nv->value) state->state |= DEFAULT;
        else state->state &= ~DEFAULT;
        update_prefsize(btn);
        return 1;
    }
    return 0;
}

static int get_handler(void* btn, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(btn, "-state");
    if (!strcmp(nv->name, "caption")) {
        nv->value = state->caption;
        return 1;
    } else if (!strcmp(nv->name, "bitmap")) {
        nv->value = state->bmp;
        return 1;
    } else if (!strcmp(nv->name, "type")) {
        switch (state->type) {
        case PUSH: nv->value = "push"; break;
        case TOGGLE: nv->value = "toggle"; break;
        case SELECT: nv->value = "select"; break;
        default: return 0;
        }
        return 1;
    } else if (!strcmp(nv->name, "selected") || !strcmp(nv->name, "checked")) {
        nv->value = (state->state & SELECTED) ? (void*)1 : NULL;
        return 1;
    } else if (!strcmp(nv->name, "group")) {
        nv->value = (void*)(ff_intptr_t)state->group;
        return 1;
    } else if (!strcmp(nv->name, "default")) {
        nv->value = (state->state & DEFAULT) ? (void*)1 : NULL;
        return 1;
    }
    return 0;
}

static int destroy_handler(void* btn, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(btn, "-state");
    ff_decref(FF_REFOBJ(state->bmp));
    free(state->caption);
    free(state);
    return 0;
}

static int newfont_handler(void* btn, ff_event_t* ev, void* data)
{
    update_prefsize(btn);
    return 0;
}

static int paint_handler(void* btn, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw = 0, th = 0, yofs = 0, xofs = 0, bw, bh;
    state_t* state = ff_get(btn, "-state");
    ff_area(btn, NULL, NULL, &w, &h);
    if (state->state & DEFAULT) {
        make_bitmaps();
        xofs = 9;
    }
    if (state->state & DOWN) {
        if (state->type == PUSH)
            ff_color(gc, FF_ACTIVE_COLOR);
        else
            ff_color_attr(gc, btn, "background", FF_3DFACE_COLOR);
        ff_fill(gc, 0, 0, w, h);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 0, 0, w - 2, 0);
        ff_line(gc, 0, 0, 0, h - 2);
        if (state->type == TOGGLE) {
            ff_color(gc, FF_3DLIGHT_COLOR);
            ff_line(gc, w - 1, 0, w - 1, h);
            ff_line(gc, 0, h - 1, w, h - 1);
            ff_color_attr(gc, btn, "foreground",
                ff_enabled(btn) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
        } else
            ff_color(gc, ff_enabled(btn) ? FF_ACTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
        xofs--;
        yofs--;
    } else if (state->type == TOGGLE && (state->state & SELECTED)) {
        ff_color_attr(gc, btn, "background", FF_3DFACE_COLOR);
        ff_fill(gc, 0, 0, w, h);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 0, 0, w, 0);
        ff_line(gc, 0, 0, 0, h);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 1, 1, w - 3, 1);
        ff_line(gc, 1, 1, 1, h - 3);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, w - 1, 0, w - 1, h);
        ff_line(gc, 0, h - 1, w, h - 1);
        ff_color_attr(gc, btn, "foreground",
            ff_enabled(btn) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    } else if (state->type == SELECT && (state->state & SELECTED)) {
        ff_color(gc, FF_ACTIVE_COLOR);
        ff_fill(gc, 0, 0, w, h);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 0, 0, w, 0);
        ff_line(gc, 0, 0, 0, h);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, w - 1, 0, w - 1, h);
        ff_line(gc, 0, h - 1, w, h - 1);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, w - 2, 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
        ff_color(gc, ff_enabled(btn) ? FF_ACTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    } else {
        ff_color_attr(gc, btn, "background", FF_3DFACE_COLOR);
        ff_fill(gc, 0, 0, w, h);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 0, 0, w, 0);
        ff_line(gc, 0, 0, 0, h);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, w - 1, 0, w - 1, h);
        ff_line(gc, 0, h - 1, w, h - 1);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, w - 2, 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
        ff_color_attr(gc, btn, "foreground",
            ff_enabled(btn) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    }
    if (state->bmp) {
        ff_bitmap_size(state->bmp, &bw, &bh);
        if (state->caption) ff_text_size(gc, state->caption, &tw, &th);
        th += bh + (state->caption ? 4 : 1);
        ff_draw(gc, ff_round_int(((float)w - (float)bw)/2.0f) - 1, ff_round_int(((float)h - (float)th)/2.0f), state->bmp);
        if (state->caption) ff_text(gc, ff_round_int((float)(w - tw)/2.0f) - xofs - 1, ff_round_int(((float)h + (float)th)/2.0f) - yofs, state->caption);
    } else if (state->caption) {
        ff_text_size(gc, state->caption, &tw, &th);
        ff_text(gc, ff_round_int(((float)w - (float)tw)/2.0f) - xofs - 1, ff_round_int(((float)h + (float)th)/2.0f) - yofs, state->caption);
    }
    if (state->state & DEFAULT)
        ff_draw(gc, w - 20 - yofs, ff_round_int(((float)h - 10.0f)/2.0f) - yofs,
            (state->state & DOWN) ? ok1bmp : ok0bmp);
    return 1;
}

static int press_handler(void* btn, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(btn, "-state");
    if (ev->z == 1) {
        if (state->type == SELECT) {
           ff_seti(btn, "selected", 1);
        } else {
            state->state |= DOWN;
            ff_paint(btn);
        }
        return FF_YES;
    }
    return 0;
}

static int release_handler(void* btn, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(btn, "-state");
    if (ev->z == 1 && (state->state & DOWN)) {
        int px, py;
        state->state &= ~DOWN;
        ff_paint(btn);
        ff_pointer(&px, &py);
        if (ff_pick(NULL, px, py) == btn) {
            if (state->type == TOGGLE)
                ff_seti(btn, "checked", !(state->state & SELECTED));
            else
                ff_post(btn, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
        }
        return FF_YES;
    }
    return 0;
}

static int timer_handler(void* btn, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(btn, "-state");
    state->state &= ~DOWN;
    ff_timer(btn, -1);
    ff_paint(btn);
    ff_post(btn, FF_ACTION, 0, 0, 0, NULL, FF_BUBBLE);
    return 1;
}

static int activate_handler(void* btn, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(btn, "-state");
    if (state->type != PUSH) return 0;
    state->state |= DOWN;
    ff_paint(btn);
    ff_timer(btn, 50);
    return 1;
}

ff_window_t ff_button(ff_window_t parent, const char* caption, ff_handler_proc_t action_handler, void* action_data)
{
    ff_window_t btn = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    ff_set(btn, "-state", state);
    ff_set(btn, "class", (void*)"button");
    ff_link(btn, FF_SET, set_handler, NULL);
    ff_link(btn, FF_GET, get_handler, NULL);
    if (caption) ff_set(btn, "caption", (void*)caption);
    ff_link(btn, FF_DESTROY, destroy_handler, NULL);
    ff_link(btn, FF_NEWFONT, newfont_handler, NULL);
    ff_link(btn, FF_PAINT, paint_handler, NULL);
    ff_link(btn, FF_PRESS, press_handler, NULL);
    ff_link(btn, FF_RELEASE, release_handler, NULL);
    ff_link(btn, FF_TIMER, timer_handler, NULL);
    ff_link(btn, FF_ACTIVATE, activate_handler, NULL);
    if (action_handler) ff_link(btn, FF_ACTION, action_handler, action_data);
    return btn;
}
