#define LFORMS_SOURCE_CODE
#include <string.h>
#include <lforms.h>

typedef struct _item_t
{
    char* text;
    void* data;
    unsigned flags;
    int y1, y2;
} item_t;

typedef struct _state_t
{
    item_t** item;
    int items;
    int selection;
    int itemsize;
    int vis1, vis2;
    int thaw_reveal;
    ff_sbinfo_t sb;
    unsigned custom_itemsize:1;
} state_t;

static void update_prefsize(ff_window_t lb, state_t* state)
{
    if (!ff_frozen(lb)) {
        int tw, th, w = 0, i;
        ff_font_t font = ff_font_of(lb);
        for (i=0; i < state->items; i++) {
            ff_font_text_size(font, state->item[i]->text, &tw, &th);
            if (w < tw) w = tw;
        }
        if (strcmp(ff_get(lb, "border"), "none"))
            w += 4;
        ff_prefsize(lb, w + 20, 60);
    }
}

static int set_handler(void* lb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(lb, "-state");
    if (!strcmp(nv->name, "border")) {
        update_prefsize(lb, state);
        ff_paint(lb);
        return 0;
    } else if (!strcmp(nv->name, "selection")) {
        ff_list_select(lb, (int)(ff_intptr_t)nv->value, FF_YES);
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        int i, newsel = -1;
        const char* text = nv->value;
        for (i=0; i < state->items; i++)
            if (!strcmp(text, state->item[i]->text)) {
                newsel = i;
                break;
            }
        if (newsel != -1 && newsel != state->selection)
            ff_list_select(lb, newsel, FF_YES);
        return 1;
    } else if (!strcmp(nv->name, "item-size")) {
        state->itemsize = (int)(ff_intptr_t)nv->value;
        if (state->itemsize < 1) {
            state->itemsize = 0;
            state->custom_itemsize = FF_NO;
        } else
            state->custom_itemsize = FF_YES;
        ff_paint(lb);
        return 1;
    }
    return 0;
}

static int get_handler(void* lb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(lb, "-state");
    if (!strcmp(nv->name, "selection")) {
        nv->value = (void*)(ff_intptr_t)state->selection;
        return 1;
    } else if (!strcmp(nv->name, "item-size")) {
        if (state->custom_itemsize || state->itemsize > 0)
            nv->value = (void*)(ff_intptr_t)state->itemsize;
        else {
            int height;
            ff_font_text_size(ff_font_of(lb), "A", NULL, &height);
            nv->value = (void*)(ff_intptr_t)height;
        }
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        if (state->selection < 0 || state->selection >= state->items)
            nv->value = "";
        else
            nv->value = state->item[state->selection]->text;
        return 1;
    }
    return 0;
}

static int destroy_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    for (i=0; i<state->items; i++) {
        free(state->item[i]->text);
        free(state->item[i]);
    }
    free(state->item);
    free(state);
    return 0;
}

static int newfont_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    state->itemsize = 0;
    update_prefsize(lb, state);
    return 0;
}

static int paint_handler(void* lb, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th;
    const char* borderattr = ff_get(lb, "border");
    int border = 0;
    state_t* state = ff_get(lb, "-state");
    int y, i;
    ff_color_t bg, fg, afg;

    if (state->itemsize < 1) {
        ff_text_size(gc, "A", &tw, &th);
        state->itemsize = th;
    }

    if (!strcmp(borderattr, "none"))
        border = 0;
    else if (!strcmp(borderattr, "normal"))
        border = 1;

    ff_area(lb, NULL, NULL, &w, &h);
    bg = ff_color_get(lb, "background", FF_3DFACE_COLOR);
    if (ff_enabled(lb)) {
        fg = ff_color_get(lb, "foreground", FF_3DTEXT_COLOR);
        afg = FF_ACTEXT_COLOR;
    } else {
        fg = ff_color_get(lb, "foreground", FF_3DDISABLEDTEXT_COLOR);
        afg = FF_3DDISABLEDTEXT_COLOR;
    }
    ff_color(gc, bg);

    y = -state->sb.pos;
    if (border == 1) {
        y += 2;
        ff_fill(gc, 21, 2, w - 2, h - 2);
    } else {
        ff_fill(gc, 19, 0, w, h);
    }
    state->vis1 = -1;
    for (i=0; i < state->items; i++) {
        ff_lbitempaint_t ip;
        if (y < -state->itemsize) {
            y += state->itemsize;
            continue;
        }
        if (y >= h) break;
        if (state->vis1 == -1) state->vis1 = i;
        state->vis2 = i;
        ip.gc = gc;
        ip.text = state->item[i]->text;
        ip.data = state->item[i]->data;
        ip.x = border ? 21 : 19;
        ip.y = y;
        ip.w = w - ip.x - (border ? 2 : 0);
        ip.h = state->itemsize;
        ip.flags = state->item[i]->flags;
        ip.bg = bg;
        ip.fg = fg;
        ip.abg = FF_ACTIVE_COLOR;
        ip.afg = afg;
        ff_send(lb, FF_LBITEMPAINT, i, (ip.flags & FF_LBISELECTED) ? 1 : 0, 0, &ip, FF_NOFLAGS);
        state->item[i]->y1 = y;
        state->item[i]->y2 = y + state->itemsize - 1;
        y += state->itemsize;
    }

    if (border == 1) {
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 0, 0, w - 2, 0);
        ff_line(gc, 0, 0, 0, h - 2);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, 1, 1, w - 3, 1);
        ff_line(gc, 1, 1, 1, h - 3);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, w - 1, 0, w - 1, h - 1);
        ff_line(gc, 0, h - 1, w - 1, h - 1);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_line(gc, w - 2, 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
    }

    state->sb.max = state->itemsize*state->items;
    ff_sbinfo_paint(&state->sb, lb, gc);

    return 1;
}

static void need_vis(ff_window_t lb, state_t* state)
{
    if (state->vis1 < 0) {
        ff_bitmap_t bmp = ff_bitmap(1, 1, NULL);
        ff_gc_t gc = ff_bitmap_gc(bmp);
        ff_event_t fake;
        memset(&fake, 0, sizeof(ff_event_t));
        fake.p = gc;
        ff_font(gc, ff_font_of(lb));
        paint_handler(lb, &fake, NULL);
        ff_bitmap_free(bmp);
    }
}

static int area_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    int border = 0;
    const char* borderattr = ff_get(lb, "border");
    if (!strcmp(borderattr, "none"))
        border = 0;
    else if (!strcmp(borderattr, "normal"))
        border = 1;
    state->sb.page = ev->y - (border ? 4 : 0);
    if (border)
        ff_sbinfo_place(&state->sb, lb, 1, 1, 20, ev->y - 2);
    else
        ff_sbinfo_place(&state->sb, lb, 0, 0, 20, ev->y);
    ff_sbinfo_update(&state->sb, lb);
    return 0;
}

static int press_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    ff_focus(lb);
    if (ff_sbinfo_press(&state->sb, lb, ev->x, ev->y, ev->z)) {
        if (ff_sbinfo_needs_timer(&state->sb))
            ff_timer(lb, 5);
        return 1;
    }
    if (ev->z == 1) {
        int index = ff_list_pick(lb, ev->x, ev->y);
        if (index != -1) ff_list_select(lb, index, FF_YES);
        return FF_YES;
    }
    return 0;
}

static int release_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    int r = ff_sbinfo_release(&state->sb, lb, ev->x, ev->y, ev->z);
    if (r && !ff_sbinfo_needs_timer(&state->sb))
        ff_timer(lb, FF_REMOVE);
    return r;
}

static int motion_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    return ff_sbinfo_motion(&state->sb, lb, ev->x, ev->y);
}

static int keydown_handler(void* lb, ff_event_t* ev, void* data)
{
    int i;
    state_t* state = ff_get(lb, "-state");
    if (!ff_focused(lb)) return 0;

    switch (ev->x) {
    case FFK_HOME:
    case FFK_LEFT:
        if (state->items > 0)
            ff_list_select(lb, 0, FF_YES);
        return 1;
    case FFK_END:
    case FFK_RIGHT:
        if (state->items > 0)
            ff_list_select(lb, state->items - 1, FF_YES);
        return 1;
    case FFK_PAGEUP:
        i = state->selection - (state->vis2 - state->vis1 - 1);
        if (i < 0) i = 0;
        ff_list_select(lb, i, FF_YES);
        return 1;
    case FFK_PAGEDOWN:
        i = state->selection + (state->vis2 - state->vis1 - 1);
        if (i >= state->items) i = state->items - 1;
        ff_list_select(lb, i, FF_YES);
        return 1;
    case FFK_UP:
        if (state->selection > 0)
            ff_list_select(lb, state->selection - 1, FF_YES);
        return 1;
    case FFK_DOWN:
        if (state->selection < state->items - 1)
            ff_list_select(lb, state->selection + 1, FF_YES);
        return 1;
    case FFK_RETURN:
        if (state->selection > -1)
            ff_post(lb, FF_ACTION, 0, 0, state->selection, NULL, FF_BUBBLE);
        return 0;
    }

    return 0;
}

static int dblclick_handler(void* lb, ff_event_t* ev, void* data)
{
    if (ev->x > 20 && ev->z == 1) {
        state_t* state = ff_get(lb, "-state");
        int index = ff_list_pick(lb, ev->x, ev->y);
        if (state->selection == index) {
            ff_post(lb, FF_ACTION, 0, 0, state->selection, NULL, FF_BUBBLE);
            return 1;
        }
    }
    return 0;
}

static int timer_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    return ff_sbinfo_timer(&state->sb, lb);
}

static int thaw_handler(void* lb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lb, "-state");
    update_prefsize(lb, state);
    if (state->thaw_reveal != -1) {
        ff_list_reveal(lb, state->thaw_reveal);
        state->thaw_reveal = -1;
    }
    return 0;
}

static int itempaint_handler(void* lb, ff_event_t* ev, void* data)
{
    ff_lbitempaint_t* ip = ev->p;
    ff_color(ip->gc, ev->y ? ip->abg : ip->bg);
    ff_fill(ip->gc, ip->x, ip->y, ip->x + ip->w, ip->y + ip->h);
    ff_color(ip->gc, ev->y ? ip->afg : ip->fg);
    ff_text(ip->gc, ip->x + 2, ip->y + ip->h, ip->text);
    return 1;
}

ff_window_t ff_list(ff_window_t parent, const char** item, void** data, int items, int selection)
{
    ff_window_t lb = ff_window(parent, 0, 0, 150, 150, FF_DOUBLEBUFFER);
    state_t* state = calloc(1, sizeof(state_t));
    ff_sbinfo_init(&state->sb, FF_SBVERTICAL);
    state->vis1 = state->vis2 = -1;
    state->thaw_reveal = -1;
    ff_set(lb, "-state", state);
    ff_set(lb, "class", (void*)"list");
    ff_set(lb, "border", "normal");
    ff_set(lb, "selection-type", "single");
    ff_link(lb, FF_SET, set_handler, NULL);
    ff_link(lb, FF_GET, get_handler, NULL);
    ff_link(lb, FF_DESTROY, destroy_handler, NULL);
    ff_link(lb, FF_NEWFONT, newfont_handler, NULL);
    ff_link(lb, FF_PAINT, paint_handler, NULL);
    ff_link(lb, FF_AREA, area_handler, NULL);
    ff_link(lb, FF_PRESS, press_handler, NULL);
    ff_link(lb, FF_RELEASE, release_handler, NULL);
    ff_link(lb, FF_MOTION, motion_handler, NULL);
    ff_link(lb, FF_KEYDOWN, keydown_handler, NULL);
    ff_link(lb, FF_DBLCLICK, dblclick_handler, NULL);
    ff_link(lb, FF_TIMER, timer_handler, NULL);
    ff_link(lb, FF_THAW, thaw_handler, NULL);
    ff_link(lb, FF_LBITEMPAINT, itempaint_handler, NULL);
    update_prefsize(lb, state);
    if (items > 0 && (item || data)) ff_list_set(lb, item, data, items, selection);

    return lb;
}

void ff_list_clear(ff_window_t lb)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    if (!state->items) return;
    for (i=0; i<state->items; i++) {
        free(state->item[i]->text);
        free(state->item[i]);
    }
    free(state->item);
    state->item = NULL;
    state->items = 0;
    state->selection = -1;
    state->sb.pos = state->sb.max = 0;
    update_prefsize(lb, state);
    ff_paint(lb);
    ff_post(lb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}

void ff_list_set(ff_window_t lb, const char** item, void** data, int items, int selection)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    if (items < 0) items = 0;
    if (state->items) {
        for (i=0; i<state->items; i++) {
            free(state->item[i]->text);
            free(state->item[i]);
        }
        free(state->item);
        state->item = NULL;
    }
    state->items = items;
    if (items) {
        state->item = malloc(sizeof(item_t*)*(size_t)items);
        for (i=0; i < items; i++) {
            state->item[i] = malloc(sizeof(item_t));
            state->item[i]->text = ff_strdup(item[i]);
            state->item[i]->data = data ? data[i] : NULL;
            state->item[i]->flags = i == selection ? FF_LBISELECTED : 0;
        }
    }
    state->selection = selection < 0 ? -1 : selection;
    if (state->selection > items - 1) state->selection = items - 1;
    update_prefsize(lb, state);
    ff_paint(lb);
    ff_post(lb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}

int ff_list_add(ff_window_t lb, const char* item, void* data)
{
    state_t* state = ff_get(lb, "-state");
    state->item = realloc(state->item, sizeof(item_t*)*(size_t)(state->items + 1));
    state->item[state->items] = malloc(sizeof(item_t));
    state->item[state->items]->text = ff_strdup(item);
    state->item[state->items]->data = data;
    state->item[state->items]->flags = 0;
    state->items++;
    state->vis1 = -1;
    update_prefsize(lb, state);
    ff_paint(lb);
    return state->items - 1;
}

void ff_list_remove(ff_window_t lb, int index)
{
    state_t* state = ff_get(lb, "-state");
    int i, oldsel = state->selection;
    int multiselect = !strcmp(ff_get(lb, "selection-type"), "multiple");
    if (index < 0 || index >= state->items) return;
    if (state->item[index]->flags & FF_LBISELECTED)
        ff_list_deselect(lb, index, FF_NO);
    free(state->item[index]->text);
    free(state->item[index]);
    state->items--;
    for (i=index; i < state->items; i++)
        state->item[i] = state->item[i + 1];
    state->vis1 = -1;
    update_prefsize(lb, state);
    ff_paint(lb);
    if (!multiselect && oldsel != -1) {
        if (oldsel >= state->items) oldsel = state->items - 1;
        ff_list_select(lb, oldsel, FF_YES);
    }
}

int ff_list_items(ff_window_t lb)
{
    state_t* state = ff_get(lb, "-state");
    return state->items;
}

int ff_list_itemindex(ff_window_t lb, const char* item)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    for (i=0; i < state->items; i++)
        if (!strcmp(state->item[i]->text, item))
            return i;
    return -1;
}

int ff_list_dataindex(ff_window_t lb, void* data)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    for (i=0; i < state->items; i++)
        if (state->item[i]->data == data)
            return i;
    return -1;
}

const char* ff_list_text(ff_window_t lb, int index)
{
    state_t* state = ff_get(lb, "-state");
    if (index >= 0 && index < state->items)
        return state->item[index]->text;
    return NULL;
}

void* ff_list_data(ff_window_t lb, int index)
{
    state_t* state = ff_get(lb, "-state");
    if (index >= 0 && index < state->items)
        return state->item[index]->data;
    return NULL;
}

void ff_list_replace(ff_window_t lb, int index, const char* text, void* data)
{
    state_t* state = ff_get(lb, "-state");
    if (index >= 0 && index < state->items) {
        free(state->item[index]->text);
        state->item[index]->text = ff_strdup(text);
        state->item[index]->data = data;
    }
}

void ff_list_select(ff_window_t lb, int index, int single)
{
    state_t* state = ff_get(lb, "-state");
    int i, multiselect, fireselect = 0, oldsel = state->selection;
    if (index >= state->items) return;
    if (index < 0) index = -1;
    multiselect = !strcmp(ff_get(lb, "selection-type"), "multiple");
    if (!multiselect) single = 1;
    if (single) {
        for (i=0; i < state->items; i++)
            if (index != i && (state->item[i]->flags & FF_LBISELECTED)) {
                state->item[i]->flags &= ~FF_LBISELECTED;
                fireselect = 1;
            }
    }
    if (index > -1) {
        if (!(state->item[index]->flags & FF_LBISELECTED)) {
            state->item[index]->flags |= FF_LBISELECTED;
            fireselect = 1;
        }
    }
    state->selection = -1;
    for (i=0; i < state->items; i++)
        if (state->item[i]->flags & FF_LBISELECTED) {
            state->selection = i;
            break;
        }
    if (state->selection != oldsel) fireselect = 1;
    if (fireselect) {
        ff_list_reveal(lb, state->selection);
        ff_post(lb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
    }
    ff_paint(lb);
}

void ff_list_deselect(ff_window_t lb, int index, int single)
{
    state_t* state = ff_get(lb, "-state");
    int i, multiselect, fireselect = 0, oldsel = state->selection;
    if (index < 0 || index >= state->items) return;
    multiselect = !strcmp(ff_get(lb, "selection-type"), "multiple");
    if (!multiselect) single = 1;
    if (single) {
        for (i=0; i < state->items; i++)
            if (index != i && (state->item[i]->flags & FF_LBISELECTED)) {
                state->item[i]->flags &= ~FF_LBISELECTED;
                fireselect = 1;
            }
    }
    if (state->item[index]->flags & FF_LBISELECTED) {
        state->item[index]->flags &= ~FF_LBISELECTED;
        fireselect = 1;
    }
    state->selection = -1;
    for (i=0; i < state->items; i++)
        if (state->item[index]->flags & FF_LBISELECTED) {
            state->selection = i;
            break;
        }
    if (state->selection != oldsel) fireselect = 1;
    if (fireselect) ff_post(lb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
    ff_paint(lb);
}

int ff_list_selected(ff_window_t lb, int index)
{
    state_t* state = ff_get(lb, "-state");
    if (index >= 0 && index < state->items)
        return (state->item[index]->flags & FF_LBISELECTED) ? 1 : 0;
    return 0;
}

int ff_list_selection(ff_window_t lb)
{
    state_t* state = ff_get(lb, "-state");
    return state->selection;
}

int ff_list_pick(ff_window_t lb, int x, int y)
{
    state_t* state = ff_get(lb, "-state");
    int i;
    if (!state->items) return -1;
    need_vis(lb, state);
    for (i=state->vis1; i <= state->vis2; i++)
        if (y >= state->item[i]->y1 && y <= state->item[i]->y2)
            return i;
    return -1;
}

int ff_list_itemarea(ff_window_t lb, int index, int* x, int* y, int* width, int* height)
{
    state_t* state = ff_get(lb, "-state");
    const char* borderattr = ff_get(lb, "border");
    int border = 0, ww, wh;
    if (index < 0 || index >= state->items) return 0;
    if (!strcmp(borderattr, "none"))
        border = 0;
    else if (!strcmp(borderattr, "normal"))
        border = 1;
    ff_area(lb, NULL, NULL, &ww, &wh);
    if (x) *x = border ? 2 : 0;
    if (y) *y = state->item[index]->y1;
    if (width) *width = ww - (border ? 4 : 0);
    if (height) *height = state->item[index]->y2 - state->item[index]->y1 + 1;
    return 1;
}

void ff_list_reveal(ff_window_t lb, int index)
{
    state_t* state = ff_get(lb, "-state");
    int y, h;
    if (index < 0 || index >= state->items) return;
    if (ff_frozen(lb)) {
        state->thaw_reveal = index;
        return;
    }
    need_vis(lb, state);
    ff_area(lb, NULL, NULL, NULL, &h);
    y = index*state->itemsize - state->sb.pos;
    if (y < 2) ff_sbinfo_set(&state->sb, lb, state->sb.pos + y);
    if (y > h - state->itemsize - 4) ff_sbinfo_set(&state->sb, lb, state->sb.pos + (y - (h - state->itemsize - 4)));
}
