#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

typedef struct _menuitem_t
{
    char* title;
    int id;
    int y1, y2;
} menuitem_t;

typedef struct _state_t
{
    menuitem_t* item;
    int items;
    int down;
    int active;
    int titley, mx, my, moving;
} state_t;

static void update_prefsize(ff_window_t menu, state_t* state)
{
    int pw = 0, ph, w, h, i;
    ff_font_t font = ff_default_font(FF_STANDARD_FONT);
    ff_font_text_size(font, "W", &w, &h);
    ph = h + 9;
    for (i=0; i<state->items; i++) {
        ff_font_text_size(font, state->item[i].title, &w, &h);
        if (w > pw) pw = w;
        ph += h + 6;
    }
    ff_prefsize(menu, pw + 9, ph);
    ff_resize(menu, pw + 9, ph);
}

static int item_by_y(state_t* state, int y)
{
    int i;
    for (i=0; i<state->items; i++)
        if (y >= state->item[i].y1 && y < state->item[i].y2)
            return i;
    return -1;
}

static int paint_handler(void* menu, ff_event_t* ev, void* data)
{
    int w, h, tw, th, y, i;
    ff_gc_t gc = ev->p;
    state_t* state = ff_get(menu, "-state");
    ff_area(menu, NULL, NULL, &w, &h);
    ff_font(gc, ff_default_font(FF_BOLD_FONT));
    ff_text_size(gc, "Menu", &tw, &th);
    ff_color(gc, FF_MENU_COLOR);
    ff_fill(gc, 0, 0, w, th + 9);
    ff_color(gc, FF_MENUTEXT_COLOR);
    ff_text(gc, 5, th + 5, "Menu");
    ff_color(gc, FF_3DFACE_COLOR);
    ff_line(gc, 1, 1, w, 1);
    ff_line(gc, 1, 1, 1, th + 8);
    ff_color(gc, FF_3DDARK_COLOR);
    ff_line(gc, w - 1, 1, w - 1, th + 8);
    ff_line(gc, 1, th + 8, w, th + 8);
    ff_line(gc, 0, 0, 0, h);
    ff_color(gc, FF_3DSHADOW_COLOR);
    ff_line(gc, w - 2, 2, w - 2, th + 7);
    ff_line(gc, 2, th + 7, w - 2, th + 7);
    state->titley = y = th + 9;
    ff_color_attr(gc, menu, "background", FF_3DFACE_COLOR);
    ff_font(gc, ff_default_font(FF_STANDARD_FONT));
    ff_fill(gc, 1, y, w, h);
    for (i=0; i<state->items; i++) {
        ff_text_size(gc, state->item[i].title, &tw, &th);
        if (i == state->active) {
            ff_color(gc, FF_3DLIGHT_COLOR);
            ff_fill(gc, 2, y, w, y + th + 5);
            ff_color_attr(gc, menu, "background", FF_3DFACE_COLOR);
        }
        state->item[i].y1 = y;
        state->item[i].y2 = y + th + 6;
        ff_color_attr(gc, menu, "foreground", FF_3DTEXT_COLOR);
        ff_text(gc, 5, y + th + 3, state->item[i].title);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 1, y, w, y);
        ff_line(gc, 1, y, 1, y + th + 5);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, w - 1, y, w - 1, y + th + 5);
        ff_line(gc, 1, y + th + 5, w, y + th + 5);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, w - 2, y + 1, w - 2, y + th + 4);
        ff_line(gc, 2, y + th + 4, w - 2, y + th + 4);
        y += th + 6;
    }

    return 1;
}

static int press_handler(void* menu, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(menu, "-state");
    if (ev->z != 1) return 1;
    ff_show(menu, FF_YES);
    if (ev->y < state->titley) {
        state->moving = 1;
        ff_pointer(&state->mx, &state->my);
    } else {
        state->active = item_by_y(state, ev->y);
        ff_paint(menu);
    }
    return 1;
}

static int release_handler(void* menu, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(menu, "-state");
    if (ev->z != 1) return 1;
    if (state->moving) {
        state->moving = 0;
        return 1;
    }
    state->down = FF_NO;
    state->active = item_by_y(state, ev->y);
    if (state->active != -1) {
        ff_post(menu, FF_ACTION, 0, 0, state->active, menu, FF_BUBBLE);
        state->active = -1;
    }
    ff_paint(menu);
    return 1;
}

static int motion_handler(void* menu, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(menu, "-state");
    int prev_active;
    if (state->moving) {
        int x, y, px, py;
        ff_area(menu, &x, &y, NULL, NULL);
        ff_pointer(&px, &py);
        ff_move(menu, (px - state->mx) + x, (py - state->my) + y);
        state->mx = px;
        state->my = py;
        return 1;
    }
    if (!state->down) return 1;
    prev_active = state->active;
    state->active = item_by_y(state, ev->y);
    if (prev_active != state->active) ff_paint(menu);
    return 1;
}

static int destroy_handler(void* win, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(win, "-state");
    int i;
    for (i=0; i < state->items; i++)
        free(state->item[i].title);
    free(state->item);
    free(state);
    return 0;
}

int ff_menu_items(ff_window_t menu)
{
    state_t* state;
    if (!menu) return 0;
    state = ff_get(menu, "-state");
    return state->items;
}

const char* ff_menu_itemtitle(ff_window_t menu, int index)
{
    state_t* state;
    if (!menu || index < 0) return NULL;
    state = ff_get(menu, "-state");
    return index < state->items ? state->item[index].title : NULL;
}

int ff_menu_itemid(ff_window_t menu, int index)
{
    state_t* state;
    if (!menu || index < 0) return 0;
    state = ff_get(menu, "-state");
    return index < state->items ? state->item[index].id : 0;
}

void ff_menu_clear(ff_window_t menu)
{
    state_t* state;
    int i;
    if (!menu) return;
    state = ff_get(menu, "-state");
    for (i=0; i<state->items; i++)
        free(state->item[i].title);
    free(state->item);
    state->item = NULL;
    state->items = 0;
    update_prefsize(menu, state);
    ff_paint(menu);
}

void ff_menu_add(ff_window_t menu, const char* title, int id)
{
    state_t* state;
    if (!menu) return;
    state = ff_get(menu, "-state");
    state->item = realloc(state->item, sizeof(menuitem_t)*(size_t)(state->items + 1));
    state->item[state->items].title = ff_strdup(title);
    state->item[state->items].id = id;
    state->items++;
    update_prefsize(menu, state);
    ff_paint(menu);
}

void ff_menu_remove(ff_window_t menu, int index)
{
    state_t* state;
    int i;
    if (!menu || index < 0) return;
    state = ff_get(menu, "-state");
    if (index >= state->items) return;
    state->items--;
    for (i=index; i<state->items; i++)
        state->item[i] = state->item[i + 1];
    state->item = realloc(state->item, sizeof(menuitem_t)*(size_t)state->items);
    update_prefsize(menu, state);
    ff_paint(menu);
}

ff_window_t ff_menu(ff_window_t parent, unsigned flags)
{
    state_t* state = calloc(1, sizeof(state_t));
    ff_window_t menu = ff_window(parent, 0, 64, 32, 32, flags);
    ff_set(menu, "-state", state);
    state->active = -1;
    ff_link(menu, FF_PAINT, paint_handler, NULL);
    ff_link(menu, FF_PRESS, press_handler, NULL);
    ff_link(menu, FF_RELEASE, release_handler, NULL);
    ff_link(menu, FF_MOTION, motion_handler, NULL);
    ff_link(menu, FF_DESTROY, destroy_handler, NULL);
    update_prefsize(menu, state);
    return menu;
}

ff_window_t ff_popup(ff_window_t parent, const char* title)
{
    ff_window_t menu;
    menu = ff_menu(parent, FF_POPUP|FF_NOFRAME|FF_AUTOHIDE|FF_TOPMOST);
    ff_retitle(menu, title);
    return menu;
}

