#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

static char* upglyph[] = {
    "16 16 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@@@@@@@@@@@@@@@.",
    "@*************-.",
    "@*************-.",
    "@******-******-.",
    "@******.******-.",
    "@*****-.-*****-.",
    "@*****...*****-.",
    "@****-...-****-.",
    "@****.....****-.",
    "@***-.....-***-.",
    "@***.......***-.",
    "@**-.......-**-.",
    "@*************-.",
    "@*************-.",
    "@--------------.",
    "................"};
static char* downglyph[] = {
    "16 16 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@@@@@@@@@@@@@@@.",
    "@*************-.",
    "@*************-.",
    "@**-.......-**-.",
    "@***.......***-.",
    "@***-.....-***-.",
    "@****.....****-.",
    "@****-...-****-.",
    "@*****...*****-.",
    "@*****-.-*****-.",
    "@******.******-.",
    "@******-******-.",
    "@*************-.",
    "@*************-.",
    "@--------------.",
    "................"};
static char* leftglyph[] = {
    "16 16 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@@@@@@@@@@@@@@@.",
    "@*************-.",
    "@*************-.",
    "@**********-**-.",
    "@********-..**-.",
    "@******-....**-.",
    "@****-......**-.",
    "@**-........**-.",
    "@****-......**-.",
    "@******-....**-.",
    "@********-..**-.",
    "@**********-**-.",
    "@*************-.",
    "@*************-.",
    "@--------------.",
    "................"};
static char* rightglyph[] = {
    "16 16 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@@@@@@@@@@@@@@@.",
    "@*************-.",
    "@*************-.",
    "@**-**********-.",
    "@**..-********-.",
    "@**....-******-.",
    "@**......-****-.",
    "@**........-**-.",
    "@**......-****-.",
    "@**....-******-.",
    "@**..-********-.",
    "@**-**********-.",
    "@*************-.",
    "@*************-.",
    "@--------------.",
    "................"};
static char* holeglyph[] = {
    "6 6 5 1\n",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    " -... ",
    "-.----",
    ".--***",
    ".-**@@",
    ".-*@@@",
    " -*@@ "};

static ff_bitmap_t upbmp, downbmp, leftbmp, rightbmp, holebmp;
static ff_bitmap_t uphibmp, downhibmp, lefthibmp, righthibmp;

static void free_bitmaps(void)
{
    if (!upbmp) return;
    ff_bitmap_free(upbmp);
    ff_bitmap_free(downbmp);
    ff_bitmap_free(leftbmp);
    ff_bitmap_free(rightbmp);
    ff_bitmap_free(holebmp);
    ff_bitmap_free(uphibmp);
    ff_bitmap_free(downhibmp);
    ff_bitmap_free(lefthibmp);
    ff_bitmap_free(righthibmp);
}

static ff_bitmap_t make_hibmp(char** xpm)
{
    char* tmp = xpm[5];
    ff_bitmap_t bmp;
    xpm[5] = "* s 3d-light";
    bmp = ff_xpm(xpm, NULL, NULL);
    xpm[5] = tmp;
    return bmp;
}

static void make_bitmaps(void)
{
    if (upbmp) return;
    upbmp = ff_xpm(upglyph, NULL, NULL);
    downbmp = ff_xpm(downglyph, NULL, NULL);
    leftbmp = ff_xpm(leftglyph, NULL, NULL);
    rightbmp = ff_xpm(rightglyph, NULL, NULL);
    holebmp = ff_xpm(holeglyph, NULL, NULL);
    uphibmp = make_hibmp(upglyph);
    downhibmp = make_hibmp(downglyph);
    lefthibmp = make_hibmp(leftglyph);
    righthibmp = make_hibmp(rightglyph);
    ff_atexit(free_bitmaps);
}

int ff_sbhelp_calc(int x, int y, int w, int h, int max, int page, int pos, unsigned flags, ff_sbhcalc_t* sb)
{
    sb->btn = (flags & FF_SBNOBUTTONS) ? 0 : 34;
    sb->sz = (flags & FF_SBHORIZONTAL) ? w : h;
    sb->trk = sb->sz - sb->btn - 4;
    if (page == FF_AUTO) page = sb->sz;
    if (page >= max || sb->trk <= 0 || max <= 0) {
        sb->tb = sb->te = -1;
        sb->ts = 0;
        return 0;
    }
    if (pos < 0) pos = 0;
    if (pos > max - page) pos = max - page;
    sb->ts = sb->trk*page/max;
    if (sb->ts < 16) sb->ts = 16;
    sb->tb = pos*(sb->trk - sb->ts)/(max - page);
    sb->te = sb->tb + sb->ts - 1;
    if (flags & FF_SBHORIZONTAL) {
        sb->tb += sb->btn;
        sb->te += sb->btn;
    }
    return 1;
}

void ff_sbhelp_draw(ff_gc_t gc, ff_window_t win, int x, int y, int w, int h, int max, int page, int pos, unsigned flags)
{
    ff_sbhcalc_t sb;
    make_bitmaps();
    ff_color(gc, FF_SCROLLBAR_BORDER_COLOR);
    ff_rect(gc, x, y, x + w - 1, y + h - 1);
    ff_color(gc, FF_SCROLLBAR_PAD_COLOR);
    ff_rect(gc, x + 1, y + 1, x + w - 2, y + h - 2);
    if (!ff_sbhelp_calc(x, y, w, h, max, page, pos, flags, &sb)) {
        ff_color(gc, FF_SCROLLBAR_COLOR);
        ff_fill(gc, x + 2, y + 2, x + w - 3, y + h - 3);
        return;
    }
    if (flags & FF_SBHORIZONTAL) {
        ff_color(gc, FF_SCROLLBAR_COLOR);
        ff_fill(gc, x + sb.btn + 2, y + 2, x + 1 + sb.tb, y + h - 3);
        ff_fill(gc, x + 3 + sb.te, y + 2, x + sb.btn + sb.trk + 1, y + h - 3);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_fill(gc, x + 3 + sb.tb, y + 3, x + sb.te, y + h - 5);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, x + 2 + sb.tb, y + 2, x + 2 + sb.tb, y + h - 4);
        ff_line(gc, x + 2 + sb.tb, y + 2, x + 1 + sb.te, y + 2);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, x + 2 + sb.te, y + 2, x + 2 + sb.te, y + h - 3);
        ff_line(gc, x + 2 + sb.tb, y + h - 3, x + 2 + sb.te, y + h - 3);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, x + 1 + sb.te, y + 3, x + 1 + sb.te, y + h - 4);
        ff_line(gc, x + 3 + sb.tb, y + h - 4, x + 1 + sb.te, y + h - 4);
        ff_draw(gc, x + 2 + sb.tb + (sb.ts-6)/2, y + 6, holebmp);
        if (sb.btn) {
            ff_color(gc, FF_SCROLLBAR_PAD_COLOR);
            ff_line(gc, x + 18, y + 2, x + 18, y + h - 3);
            ff_draw(gc, x + 2, y + 2, (FF_SBBUTTON1 & flags) ? lefthibmp : leftbmp);
            ff_line(gc, x + 35, y + 2, x + 35, y + h - 3);
            ff_draw(gc, x + 19, y + 2, (FF_SBBUTTON2 & flags) ? righthibmp : rightbmp);
        }
    } else {
        ff_color(gc, FF_SCROLLBAR_COLOR);
        ff_fill(gc, x + 2, y + 2, x + w - 3, y + 1 + sb.tb);
        ff_fill(gc, x + 2, y + 3 + sb.te, x + w - 3, y + 2 + sb.trk - 1);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_fill(gc, x + 3, y + 3 + sb.tb, x + w - 5, y + sb.te);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, x + 2, y + 2 + sb.tb, x + w - 4, y + 2 + sb.tb);
        ff_line(gc, x + 2, y + 2 + sb.tb, x + 2, y + 1 + sb.te);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, x + w - 3, y + 2 + sb.tb, x + w - 3, y + 2 + sb.te);
        ff_line(gc, x + 2, y + 2 + sb.te, x + w - 3, y + 2 + sb.te);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, x + w - 4, y + 3 + sb.tb, x + w - 4, y + 1 + sb.te);
        ff_line(gc, x + 3, y + 1 + sb.te, x + w - 4, y + 1 + sb.te);
        ff_draw(gc, x + 6, y + 2 + sb.tb + (sb.ts-6)/2, holebmp);
        if (sb.btn) {
            ff_color(gc, FF_SCROLLBAR_PAD_COLOR);
            ff_line(gc, x + 2, y + sb.trk + 2, x + w - 3, y + sb.trk + 2);
            ff_draw(gc, x + 2, y + sb.trk + 3, (FF_SBBUTTON1 & flags) ? uphibmp : upbmp);
            ff_line(gc, x + 2, y + sb.trk + 19, x + w - 3, y + sb.trk + 19);
            ff_draw(gc, x + 2, y + sb.trk + 20, (FF_SBBUTTON2 & flags) ? downhibmp : downbmp);
        }
    }
}

void ff_sbinfo_init(ff_sbinfo_t* sb, unsigned flags)
{
    memset(sb, 0, sizeof(ff_sbinfo_t));
    sb->flags = flags;
}

void ff_sbinfo_place(ff_sbinfo_t* sb, ff_window_t win, int x, int y, int w, int h)
{
    sb->x = x;
    sb->y = y;
    sb->w = w;
    sb->h = h;
    if (sb->page == FF_AUTO) ff_sbinfo_update(sb, win);
}

void ff_sbinfo_update(ff_sbinfo_t* sb, ff_window_t win)
{
    sb->pos--;
    ff_sbinfo_set(sb, win, sb->pos + 1);
}

void ff_sbinfo_paint(ff_sbinfo_t* sb, ff_window_t win, ff_gc_t gc)
{
    ff_sbhelp_draw(gc, win, sb->x, sb->y, sb->w, sb->h, sb->max, sb->page, sb->pos, sb->flags);
}

void ff_sbinfo_set(ff_sbinfo_t* sb, ff_window_t win, int pos)
{
    int page = sb->page == FF_AUTO ? ((sb->flags & FF_HORIZONTAL) ? sb->w : sb->h) : sb->page;
    if (pos > sb->max - page) pos = sb->max - page;
    if (pos < 0) pos = 0;
    if (pos != sb->pos) {
        sb->pos = pos;
        ff_post(win, FF_CHANGE, pos, sb->max, page, NULL, FF_BUBBLE);
        ff_paint(win);
    }
}

ff_sbpart_t ff_sbinfo_part(ff_sbinfo_t* sb, int x, int y)
{
    ff_sbhcalc_t sbc;
    if (!ff_sbhelp_calc(sb->x, sb->y, sb->w, sb->h, sb->max, sb->page, sb->pos, sb->flags, &sbc))
        return FF_SBNONE;
    if (sb->flags & FF_SBHORIZONTAL) {
        x -= sb->x;
        if (x <= sbc.btn/2) return FF_SBARROW1;
        if (x <= sbc.btn) return FF_SBARROW2;
        if (x < sbc.tb) return FF_SBBEFORE;
        if (x > sbc.te) return FF_SBAFTER;
        if (x >= sbc.tb && x <= sbc.te) return FF_SBTHUMB;
    } else {
        y -= sb->y;
        if (y >= sb->h - sbc.btn/2) return FF_SBARROW2;
        if (y >= sb->h - sbc.btn) return FF_SBARROW1;
        if (y < sbc.tb) return FF_SBBEFORE;
        if (y > sbc.te) return FF_SBAFTER;
        if (y >= sbc.tb && y <= sbc.te) return FF_SBTHUMB;
    }
    return FF_SBNONE;
}

int ff_sbinfo_press(ff_sbinfo_t* sb, ff_window_t win, int x, int y, int z)
{
    if (z == 1 && x >= sb->x && y >= sb->y && x < sb->x + sb->w && y < sb->y + sb->h) {
        ff_sbpart_t part = ff_sbinfo_part(sb, x, y);
        sb->capos = (sb->flags & FF_SBHORIZONTAL) ? x : y;
        sb->precapos = sb->pos;
        switch (part) {
        case FF_SBARROW1:
            sb->cap = part;
            sb->flags |= FF_SBBUTTON1;
            ff_sbinfo_set(sb, win, sb->pos - 1);
            return 1;
        case FF_SBARROW2:
            sb->cap = part;
            sb->flags |= FF_SBBUTTON2;
            ff_sbinfo_set(sb, win, sb->pos + 1);
            return 1;
        case FF_SBBEFORE:
            ff_sbinfo_set(sb, win, sb->pos - (sb->page == FF_AUTO ? ((sb->flags & FF_HORIZONTAL) ? sb->w : sb->h) : sb->page));
            return 1;
        case FF_SBAFTER:
            ff_sbinfo_set(sb, win, sb->pos + (sb->page == FF_AUTO ? ((sb->flags & FF_HORIZONTAL) ? sb->w : sb->h) : sb->page));
            return 1;
        case FF_SBTHUMB:
            sb->cap = FF_SBTHUMB;
            return 1;
        default:
            return 1;
        }
    } else if (z == 4) {
        ff_sbinfo_set(sb, win, sb->pos - (sb->page == FF_AUTO ? ((sb->flags & FF_HORIZONTAL) ? sb->w : sb->h)/2 : sb->page/2));
        return 1;
    } else if (z == 5) {
        ff_sbinfo_set(sb, win, sb->pos + (sb->page == FF_AUTO ? ((sb->flags & FF_HORIZONTAL) ? sb->w : sb->h)/2 : sb->page/2));
        return 1;
    }
    return 0;
}

int ff_sbinfo_release(ff_sbinfo_t* sb, ff_window_t win, int x, int y, int z)
{
    switch (sb->cap) {
    case FF_SBARROW1:
        sb->flags &=~FF_SBBUTTON1;
        /* FALLTHRU */ /* GCC magic comment */
    case FF_SBARROW2:
        if (sb->cap == FF_SBARROW2) sb->flags &=~FF_SBBUTTON2;
        /* FALLTHRU */ /* GCC magic comment */
    case FF_SBBEFORE:
    case FF_SBAFTER:
    case FF_SBTHUMB:
        sb->cap = FF_SBNONE;
        ff_paint(win);
        return 1;
    default:
        return 0;
    }
}

int ff_sbinfo_motion(ff_sbinfo_t* sb, ff_window_t win, int x, int y)
{
    if (sb->cap == FF_SBTHUMB) {
        int pos = (sb->flags & FF_SBHORIZONTAL) ? x : y;
        int delta = pos - sb->capos;
        int page;
        ff_sbhcalc_t h;
        if (!ff_sbhelp_calc(sb->x, sb->y, sb->w, sb->h, sb->max, sb->page, sb->pos, sb->flags, &h))
            return 0;
        page = sb->page == FF_AUTO ? h.sz : sb->page;
        ff_sbinfo_set(sb, win, sb->precapos + delta*(sb->max - page)/(h.trk - h.ts));
        return 1;
    }
    return 0;
}

int ff_sbinfo_needs_timer(ff_sbinfo_t* sb)
{
    return sb->cap == FF_SBARROW1 || sb->cap == FF_SBARROW2;
}

int ff_sbinfo_timer(ff_sbinfo_t* sb, ff_window_t win)
{
    switch (sb->cap) {
    case FF_SBARROW1:
        ff_sbinfo_set(sb, win, sb->pos - 1);
        return 1;
    case FF_SBARROW2:
        ff_sbinfo_set(sb, win, sb->pos + 1);
        return 1;
    default:
        return 0;
    }
}

typedef struct _state_t
{
    ff_sbinfo_t i;
} state_t;

static int set_handler(void* sb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(sb, "-state");
    if (!strcmp(nv->name, "position")) {
        ff_sbinfo_set(&state->i, sb, (int)(ff_intptr_t)nv->value);
        return 1;
    } else if (!strcmp(nv->name, "max")) {
        int max = (int)(ff_intptr_t)nv->value;
        if (max == state->i.max) return 1;
        if (max < 0) max = 0;
        state->i.max = max;
        state->i.pos--;
        ff_sbinfo_set(&state->i, sb, state->i.pos + 1);
        return 1;
    } else if (!strcmp(nv->name, "page-size")) {
        int page = (int)(ff_intptr_t)nv->value;
        if (page < 0) page = FF_AUTO;
        if (page == state->i.page) return 1;
        state->i.page = page;
        state->i.pos--;
        ff_sbinfo_set(&state->i, sb, state->i.pos + 1);
        return 1;
    }
    return 0;
}

static int get_handler(void* sb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(sb, "-state");
    if (!strcmp(nv->name, "position")) {
        nv->value = (void*)(ff_intptr_t)state->i.pos;
        return 1;
    } else if (!strcmp(nv->name, "max")) {
        nv->value = (void*)(ff_intptr_t)state->i.max;
        return 1;
    } else if (!strcmp(nv->name, "page-size")) {
        nv->value = (void*)(ff_intptr_t)state->i.page;
        return 1;
    } else if (!strcmp(nv->name, "orientation")) {
        if (state->i.flags & FF_SBHORIZONTAL)
            nv->value = "horizontal";
        else
            nv->value = "vertical";
        return 1;
    }
    return 0;
}

static int destroy_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    free(state);
    return 0;
}

static int paint_handler(void* sb, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h;
    state_t* state = ff_get(sb, "-state");
    ff_area(sb, NULL, NULL, &w, &h);
    ff_sbinfo_paint(&state->i, sb, gc);
    return 1;
}

static int area_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    ff_sbinfo_place(&state->i, sb, 0, 0, ev->x, ev->y);
    return 0;
}

static int press_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    int r = ff_sbinfo_press(&state->i, sb, ev->x, ev->y, ev->z);
    if (r && ff_sbinfo_needs_timer(&state->i))
        ff_timer(sb, 50);
    return r;
}

static int release_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    int r = ff_sbinfo_release(&state->i, sb, ev->x, ev->y, ev->z);
    if (r && !ff_sbinfo_needs_timer(&state->i))
        ff_timer(sb, FF_REMOVE);
    return r;
}

static int motion_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    return ff_sbinfo_motion(&state->i, sb, ev->x, ev->y);
}

static int timer_handler(void* sb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sb, "-state");
    ff_sbinfo_timer(&state->i, sb);
    return 1;
}

ff_window_t ff_scrollbar(ff_window_t parent, int pagesize, int max, int position, int horizontal)
{
    ff_window_t sb = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    ff_sbinfo_init(&state->i, horizontal ? FF_SBHORIZONTAL : FF_SBVERTICAL);
    if (max < 0) max = 0;
    if (position < 0) position = 0;
    if (position > max) position = max;
    if (pagesize < 0) pagesize = FF_AUTO;
    if (pagesize > max) pagesize = max;
    state->i.page = pagesize;
    state->i.max = max;
    state->i.pos = -1;
    ff_set(sb, "-state", state);
    ff_set(sb, "class", (void*)"scrollbar");
    ff_set(sb, "background", (void*)FF_SCROLLBAR_COLOR);
    ff_link(sb, FF_SET, set_handler, NULL);
    ff_link(sb, FF_GET, get_handler, NULL);
    ff_link(sb, FF_DESTROY, destroy_handler, NULL);
    ff_link(sb, FF_PAINT, paint_handler, NULL);
    ff_link(sb, FF_PRESS, press_handler, NULL);
    ff_link(sb, FF_AREA, area_handler, NULL);
    ff_link(sb, FF_RELEASE, release_handler, NULL);
    ff_link(sb, FF_MOTION, motion_handler, NULL);
    ff_link(sb, FF_TIMER, timer_handler, NULL);
    ff_prefsize(sb, horizontal ? 128 : 20, horizontal ? 20 : 128);
    ff_sbinfo_place(&state->i, sb, 0, 0, horizontal ? 128 : 20, horizontal ? 20 : 128);
    ff_sbinfo_set(&state->i, sb, position);
    return sb;
}
