unit GLTextDrawers;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Types, Graphics, gl, GLext;

type

  { TGLTextDrawer }

  TGLTextDrawer = class(TComponent)
  private
    Buffer: TBitmap;
    FViewportSize: TSize;
    function GetFont: TFont;
    procedure NeedBuffer;
    procedure SetFont(AValue: TFont);
  public
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure Draw(X, Y: Integer; Text: string; const Color: TColor; const Outline: TColor=clNone; const Background: TColor=clNone);
    procedure DrawAligned(X, Y: Integer; Text: string; AX, AY: Double; const Color: TColor; const Outline: TColor=clNone; const Background: TColor=clNone);
    function GetTextWidth(AText: string): Integer; inline;
    function GetTextHeight(AText: string): Integer; inline;
    function GetTextSize(AText: string): TSize; inline;
    property ViewportSize: TSize read FViewportSize write FViewportSize;
  published
    property Font: TFont read GetFont write SetFont;
  end;

procedure Register;

implementation

uses
  LResources, Forms;

procedure Register;
begin
  {$I gltextdrawers_icon.lrs}
  RegisterComponents('RTTK', [TGLTextDrawer]);
end;

{ TGLTextDrawer }

procedure TGLTextDrawer.NeedBuffer;
begin
  if not Assigned(Buffer) then begin
    Buffer:=TBitmap.Create;
    Buffer.PixelFormat:=pf32bit;
    Buffer.Width:=1;
    Buffer.Height:=1;
  end;
end;

function TGLTextDrawer.GetFont: TFont;
begin
  NeedBuffer;
  Result:=Buffer.Canvas.Font;
end;

procedure TGLTextDrawer.SetFont(AValue: TFont);
begin
  NeedBuffer;
  Buffer.Canvas.Font:=AValue;
end;

procedure TGLTextDrawer.AfterConstruction;
begin
  inherited AfterConstruction;
  Font:=Screen.SystemFont;
end;

procedure TGLTextDrawer.BeforeDestruction;
begin
  if Assigned(Buffer) then Buffer.Free;
  inherited BeforeDestruction;
end;

procedure TGLTextDrawer.Draw(X, Y: Integer; Text: string; const Color: TColor;
  const Outline: TColor; const Background: TColor);
var
  W, H: Integer;
  Line: Integer;
  Data: PByte;
  Style: TTextStyle;
begin
  NeedBuffer;
  if not Assigned(glWindowPos2i) then begin
    if not Load_GL_version_1_4 then Exit;
  end;
  Buffer.Canvas.GetTextSize(Text, W{%H-}, H{%H-});
  if Outline <> clNone then begin
    Inc(W, 2);
    Inc(H, 2);
    Dec(X);
    Dec(Y);
  end;
  if (X > ViewportSize.cx) or (Y > ViewportSize.cy) or
     (X < -W) or (Y < -H) then Exit;
  if Buffer.Width < W then Buffer.Width:=W;
  if Buffer.Height < H then Buffer.Height:=H;
  if Background=clNone then begin
    Buffer.BeginUpdate;
    Data:=Buffer.RawImage.Data;
    for Line:=0 to H - 1 do begin
      {$IFDEF WINDOWS}
      glReadPixels(X, ViewportSize.cy - (Y + Line) - 1, W, 1, GL_BGRA, GL_UNSIGNED_BYTE, Data);
      {$ELSE}
      glReadPixels(X, ViewportSize.cy - (Y + Line) - 1, W, 1, GL_RGBA, GL_UNSIGNED_BYTE, Data);
      {$ENDIF}
      Data:=Data + Buffer.RawImage.Description.BytesPerLine
    end;
    Buffer.EndUpdate;
  end else begin
    Buffer.Canvas.Pen.Color:=Background;
    Buffer.Canvas.Brush.Color:=Background;
    Buffer.Canvas.Brush.Style:=bsSolid;
    Buffer.Canvas.Rectangle(0, 0, W + 4, H + 4);
  end;
  Style:=Buffer.Canvas.TextStyle;
  Style.Opaque:=False;
  Style.ExpandTabs:=False;
  Style.ShowPrefix:=False;
  Style.SingleLine:=True;
  Buffer.Canvas.TextStyle:=Style;
  Buffer.Canvas.Brush.Style:=bsClear;
  if Outline=clNone then begin
    Buffer.Canvas.Font.Color:=Color;
    Buffer.Canvas.TextOut(0, 0, Text);
  end else begin
    Buffer.Canvas.Font.Color:=Outline;
    Buffer.Canvas.TextOut(0, 0, Text);
    Buffer.Canvas.TextOut(1, 0, Text);
    Buffer.Canvas.TextOut(2, 0, Text);
    Buffer.Canvas.TextOut(0, 1, Text);
    Buffer.Canvas.TextOut(2, 1, Text);
    Buffer.Canvas.TextOut(0, 2, Text);
    Buffer.Canvas.TextOut(1, 2, Text);
    Buffer.Canvas.TextOut(2, 2, Text);
    Buffer.Canvas.Font.Color:=Color;
    Buffer.Canvas.TextOut(1, 1, Text);
  end;
  Buffer.BeginUpdate;
  Data:=Buffer.RawImage.Data;
  for Line:=0 to H - 1 do begin
    glWindowPos2i(X, ViewportSize.cy - (Y + Line) - 1);
    {$IFDEF WINDOWS}
    glDrawPixels(W, 1, GL_BGRA, GL_UNSIGNED_BYTE, Data);
    {$ELSE}
    glDrawPixels(W, 1, GL_RGBA, GL_UNSIGNED_BYTE, Data);
    {$ENDIF}
    Data:=Data + Buffer.RawImage.Description.BytesPerLine;
  end;
  Buffer.EndUpdate;
end;

procedure TGLTextDrawer.DrawAligned(X, Y: Integer; Text: string; AX,
  AY: Double; const Color: TColor; const Outline: TColor;
  const Background: TColor);
var
  W, H: Integer;
begin
  NeedBuffer;
  Buffer.Canvas.GetTextSize(Text, W{%H-}, H{%H-});
  Draw(X - Round(W*AX), Y - Round(H*AY), Text, Color, Outline, Background);
end;

function TGLTextDrawer.GetTextWidth(AText: string): Integer;
begin
  NeedBuffer;
  Result:=Buffer.Canvas.GetTextWidth(AText);
end;

function TGLTextDrawer.GetTextHeight(AText: string): Integer;
begin
  NeedBuffer;
  Result:=Buffer.Canvas.GetTextHeight(AText);
end;

function TGLTextDrawer.GetTextSize(AText: string): TSize;
begin
  NeedBuffer;
  Result.cx:=0;
  Result.cy:=0;
  Buffer.Canvas.GetTextSize(AText, Result.cx, Result.cy);
end;

end.
