unit GradientSliders;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, LCLType;

type
  TGradientType = (gtTwoColors, gtCustomPaint);
  TShowGradientValue = (svNone, svValue, svPercent, svCustom);
  TOnPaintGradient = procedure(Sender: TObject; ACanvas: TCanvas; Coord: Integer; AValue: Single) of object;
  TOnGradientValueText = procedure(Sender: TObject; AValue: Single; var ValueText: string) of object;

  { TGradientSlider }

  TGradientSlider = class(TGraphicControl)
  private
    FAutoTextColor: Boolean;
    FOnChanged: TNotifyEvent;
    FOnGradientValueText: TOnGradientValueText;
    FShowGradientValue: TShowGradientValue;
    FStartColor: TColor;
    FEndColor: TColor;
    FGradientType: TGradientType;
    FTextOutline: Boolean;
    FTextOutlineColor: TColor;
    FValue: Single;
    FOnPaintGradient: TOnPaintGradient;
    Sliding: Boolean;
    function GetColorValue: TColor;
    procedure SetAutoTextColor(AValue: Boolean);
    procedure SetEndColor(AValue: TColor);
    procedure SetGradientType(AValue: TGradientType);
    procedure SetOnPaintGradient(AValue: TOnPaintGradient);
    procedure SetShowGradientValue(AValue: TShowGradientValue);
    procedure SetStartColor(AValue: TColor);
    procedure SetTextOutline(AValue: Boolean);
    procedure SetTextOutlineColor(AValue: TColor);
    procedure SetValue(AValue: Single);
  protected
    procedure RealSetText(const Value: TCaption); override;
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    property ColorValue: TColor read GetColorValue;
  published
    // Common properties
    property Align;
    property Anchors;
    property BorderSpacing;
    property Caption;
    property Constraints;
    property Cursor;
    property Enabled;
    property Font;
    property Hint;
    property ShowHint;
    property Width default 100;
    property Height default 13;
    // Own properties
    property StartColor: TColor read FStartColor write SetStartColor default clRed;
    property EndColor: TColor read FEndColor write SetEndColor default clBlue;
    property Value: Single read FValue write SetValue default 0;
    property GradientType: TGradientType read FGradientType write SetGradientType default gtTwoColors;
    property AutoTextColor: Boolean read FAutoTextColor write SetAutoTextColor default True;
    property TextOutline: Boolean read FTextOutline write SetTextOutline default True;
    property TextOutlineColor: TColor read FTextOutlineColor write SetTextOutlineColor default clBlack;
    property ShowGradientValue: TShowGradientValue read FShowGradientValue write SetShowGradientValue default svNone;
    // Own events
    property OnChanged: TNotifyEvent read FOnChanged write FOnChanged;
    property OnPaintGradient: TOnPaintGradient read FOnPaintGradient write SetOnPaintGradient;
    property OnGradientValueText: TOnGradientValueText read FOnGradientValueText write FOnGradientValueText;
  end;

procedure Register;

implementation

uses LCLIntf, MiscUtils;

procedure Register;
begin
  {$I gradientsliders_icon.lrs}
  RegisterComponents('RTTK',[TGradientSlider]);
end;

{ TGradientSlider }

function TGradientSlider.GetColorValue: TColor;
var
  R1, G1, B1, R2, G2, B2: Byte;
begin
  RedGreenBlue(ColorToRGB(StartColor), R1, G1, B1);
  RedGreenBlue(ColorToRGB(EndColor), R2, G2, B2);
  Result:=RGBToColor(
    Round(R1*(1-Value) + R2*Value),
    Round(G1*(1-Value) + G2*Value),
    Round(B1*(1-Value) + B2*Value));
end;

procedure TGradientSlider.SetAutoTextColor(AValue: Boolean);
begin
  if FAutoTextColor=AValue then Exit;
  FAutoTextColor:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetEndColor(AValue: TColor);
begin
  if FEndColor=AValue then Exit;
  FEndColor:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetGradientType(AValue: TGradientType);
begin
  if FGradientType=AValue then Exit;
  FGradientType:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetOnPaintGradient(AValue: TOnPaintGradient);
begin
  if FOnPaintGradient=AValue then Exit;
  FOnPaintGradient:=AValue;
  if GradientType=gtCustomPaint then Invalidate;
end;

procedure TGradientSlider.SetShowGradientValue(AValue: TShowGradientValue);
begin
  if FShowGradientValue=AValue then Exit;
  FShowGradientValue:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetStartColor(AValue: TColor);
begin
  if FStartColor=AValue then Exit;
  FStartColor:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetTextOutline(AValue: Boolean);
begin
  if FTextOutline=AValue then Exit;
  FTextOutline:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetTextOutlineColor(AValue: TColor);
begin
  if FTextOutlineColor=AValue then Exit;
  FTextOutlineColor:=AValue;
  Invalidate;
end;

procedure TGradientSlider.SetValue(AValue: Single);
begin
  if AValue < 0 then AValue:=0 else if AValue > 1 then AValue:=1;
  if FValue=AValue then Exit;
  FValue:=AValue;
  Invalidate;
  if Assigned(OnChanged) then OnChanged(Self);
end;

procedure TGradientSlider.RealSetText(const Value: TCaption);
begin
  inherited RealSetText(Value);
  Invalidate;
end;

procedure TGradientSlider.Paint;

  procedure PaintTheGradient(Canvas: TCanvas; X1, Y1, X2, Y2: Integer);
  var
    X: Integer;
    R1, G1, B1, R2, G2, B2: Byte;
    HereValue: Extended;
  begin
    case GradientType of
      gtTwoColors: begin
        RedGreenBlue(ColorToRGB(StartColor), R1, G1, B1);
        RedGreenBlue(ColorToRGB(EndColor), R2, G2, B2);
        for X:=X1 to X2 do begin
          HereValue:=(X - X1)/(X2 - X1);
          Canvas.Pen.Color:=RGBToColor(
            Round(R1*(1-HereValue) + R2*HereValue),
            Round(G1*(1-HereValue) + G2*HereValue),
            Round(B1*(1-HereValue) + B2*HereValue));
          Canvas.Line(X, Y1, X, Y2 + 1);
        end;
      end;
      gtCustomPaint: if Assigned(OnPaintGradient) then begin
        for X:=X1 to X2 do
          OnPaintGradient(Self, Canvas, X, (X - X1)/(X2 - X1));
      end;
    end;
  end;

  procedure PaintTheThumb(Canvas: TCanvas);
  var
    X: Integer;
    R: Byte;
    G: Byte;
    B: Byte;
    Col: TColor;
  begin
    X:=1 + Round(Value*(Width - 3));
    RedGreenBlue(Canvas.Pixels[X, Height div 2 + 1], R, G, B);
    if R*0.21 + G*0.72 + B*0.07 < 120 then Col:=clWhite else Col:=clBlack;
    Canvas.Pixels[X - 1, 1]:=Col;
    Canvas.Pixels[X, 1]:=Col;
    Canvas.Pixels[X, 2]:=Col;
    Canvas.Pixels[X + 1, 1]:=Col;
    Canvas.Pixels[X - 1, Height - 2]:=Col;
    Canvas.Pixels[X, Height - 2]:=Col;
    Canvas.Pixels[X, Height - 3]:=Col;
    Canvas.Pixels[X + 1, Height - 2]:=Col;
    Canvas.Pixels[X, Height div 2]:=Col;
  end;

var
  TheTextStyle: TTextStyle;
  Backbuffer: TBitmap;
  VisibleText: TCaption;
begin
  Backbuffer:=TBitmap.Create;
  Backbuffer.SetSize(Width, Height);
  with Backbuffer.Canvas do begin
    PaintTheGradient(Backbuffer.Canvas, 1, 1, Self.Width - 2, Self.Height - 2);
    PaintTheThumb(Backbuffer.Canvas);
    Pen.Color:=cl3DShadow;
    Line(0, 0, Width, 0);
    Line(0, 0, 0, Height);
    Pen.Color:=cl3DHiLight;
    Line(Self.Width - 1, 0, Self.Width - 1, Self.Height);
    Line(0, Self.Height - 1, Self.Width - 1, Self.Height - 1);
    if Caption <> '' then VisibleText:=Caption else VisibleText:='';
    case ShowGradientValue of
      svNone: begin end;
      svValue, svCustom: begin
        VisibleText:=Format('%0.2f', [Value], NormalFormatSettings);
        if Assigned(OnGradientValueText) and (ShowGradientValue=svCustom) then
          OnGradientValueText(Self, Value, VisibleText);
        if Caption <> '' then
          VisibleText:=Caption + ' ' + VisibleText;
      end;
      svPercent: begin
        if VisibleText <> '' then
          VisibleText:=VisibleText + ' ' + IntToStr(Round(Value*100)) + '%'
        else
          VisibleText:=IntToStr(Round(Value*100)) + '%';
      end;
    end;
    if VisibleText <> '' then begin
      TheTextStyle:=TextStyle;
      TheTextStyle.Clipping:=True;
      TheTextStyle.Opaque:=False;
      TheTextStyle.Alignment:=taCenter;
      TheTextStyle.Layout:=tlCenter;
      if TextOutline then begin
        Font.Color:=TextOutlineColor;
        TextRect(Rect(0, 0, Self.Width - 3, Self.Height - 3), 0, 0, VisibleText, TheTextStyle);
        TextRect(Rect(2, 0, Self.Width - 1, Self.Height - 3), 2, 0, VisibleText, TheTextStyle);
        TextRect(Rect(0, 2, Self.Width - 3, Self.Height - 1), 0, 0, VisibleText, TheTextStyle);
        TextRect(Rect(2, 2, Self.Width - 1, Self.Height - 1), 2, 0, VisibleText, TheTextStyle);
        TextRect(Rect(0, 1, Self.Width - 3, Self.Height - 2), 0, 1, VisibleText, TheTextStyle);
        TextRect(Rect(2, 1, Self.Width - 1, Self.Height - 2), 2, 1, VisibleText, TheTextStyle);
        TextRect(Rect(1, 0, Self.Width - 2, Self.Height - 3), 1, 0, VisibleText, TheTextStyle);
        TextRect(Rect(1, 2, Self.Width - 2, Self.Height - 1), 1, 2, VisibleText, TheTextStyle);
      end;
      if AutoTextColor then Font.Color:=ColorValue else Font.Color:=Self.Font.Color;
      TextRect(Rect(1, 1, Self.Width - 2, Self.Height - 2), 1, 1, VisibleText, TheTextStyle);
    end;
  end;
  Canvas.Draw(0, 0, Backbuffer);
  Backbuffer.Free;
end;

procedure TGradientSlider.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  if Button=mbLeft then begin
    Value:=(X - 1)/(Width - 3);
    Sliding:=True;
    Exit;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TGradientSlider.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseMove(Shift, X, Y);
  if Sliding then Value:=(X - 1)/(Width - 3);
end;

procedure TGradientSlider.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
begin
  if Button=mbLeft then Sliding:=False;
  inherited MouseUp(Button, Shift, X, Y);
end;

constructor TGradientSlider.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:=100;
  Height:=13;
  FStartColor:=clRed;
  FEndColor:=clBlue;
  FGradientType:=gtTwoColors;
  FValue:=0;
  Font.Color:=clWhite;
  FTextOutline:=True;
  FTextOutlineColor:=clBlack;
end;

end.
