unit Nodes;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FGL;

type
  EInvalidNodeOperation = exception;

  TNode = class;
  TNodeList = specialize TFPGObjectList<TNode>;
  TNodeArray = array of TNode;
  TNodeClass = class of TNode;

  TNodeFlag = (nfModified);
  TNodeFlags = set of TNodeFlag;

  { TNode }

  TNode = class
  private
    FChildren: TNodeList;
    FName: string;
    FNodeFlags: TNodeFlags;
    FParent: TNode;
    function GetChildCount: Integer; inline;
    function GetChildren(AIndex: Integer): TNode; inline;
    procedure SetName(AValue: string);
    procedure SetParent(AValue: TNode);
  protected
    procedure ParentChanged; virtual;
    procedure ChildAdded(AChild: TNode); virtual;
    procedure ChildRemoved(AChild: TNode); virtual;
    procedure ChildrenChanged; virtual;
    procedure NameChanged(OldName: string); virtual;
    procedure ModifyChanged; virtual;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    function CanAddChild(AChild: TNode): Boolean; virtual;
    procedure Add(AChild: TNode); inline;
    procedure Remove(AChild: TNode); inline;
    procedure Delete(AChild: TNode); inline;
    procedure Clear; inline;
    procedure Swap(AChild1, AChild2: TNode);
    procedure Swap(Index1, Index2: Integer);
    function IndexOf(AChild: TNode): Integer; inline;
    function HasChild(AChild: TNode): Boolean; inline;
    function FindChild(AName: string; Deep: Boolean=True): TNode;
    function HasChildren: Boolean; inline;
    function IsEmpty: Boolean; inline;
    function GetChildByPath(APath: string; Separator: Char='/'): TNode;
    function GetPath(Separator: Char='/'): string;
    function GetUniqueChildName(Prefix: string; Deep: Boolean=True): string;
    function CanModify: Boolean; virtual;
    function MarkModified: Boolean;
    procedure ClearModified;
    function IsModified: Boolean; inline;
    function GetRootNode: TNode;
    function GetAncestorWithClass(AClass: TNodeClass): TNode;
    function GetChildrenWithClass(AClass: TNodeClass; Deep: Boolean=True): TNodeArray;
    property Parent: TNode read FParent write SetParent;
    property Name: string read FName write SetName;
    property Children[AIndex: Integer]: TNode read GetChildren;
    property ChildCount: Integer read GetChildCount;
    property NodeFlags: TNodeFlags read FNodeFlags;
  end;

implementation

{ TNode }

function TNode.GetChildCount: Integer;
begin
  Result:=FChildren.Count;
end;

function TNode.GetChildren(AIndex: Integer): TNode;
begin
  Result:=FChildren[AIndex];
end;

procedure TNode.SetName(AValue: string);
var
  OldName: String;
begin
  if FName=AValue then Exit;
  OldName:=FName;
  FName:=AValue;
  NameChanged(OldName);
end;

procedure TNode.SetParent(AValue: TNode);
begin
  if FParent=AValue then Exit;
  if Assigned(FParent) then FParent.Remove(Self);
  if Assigned(AValue) then
    AValue.Add(Self)
  else begin
    FParent:=nil;
    ParentChanged;
  end;
end;

procedure TNode.ParentChanged;
begin
end;

procedure TNode.ChildAdded(AChild: TNode);
begin
end;

procedure TNode.ChildRemoved(AChild: TNode);
begin
end;

procedure TNode.ChildrenChanged;
begin
end;

procedure TNode.NameChanged(OldName: string);
begin
end;

procedure TNode.ModifyChanged;
begin
end;

constructor TNode.Create;
begin
  FChildren:=TNodeList.Create(False);
end;

destructor TNode.Destroy;
var
  I: Integer;
begin
  if Assigned(FParent) then FParent.Remove(Self);
  for I:=ChildCount - 1 downto 0 do begin
    FChildren[I].FParent:=nil;
    FChildren[I].Free;
  end;
  FreeAndNil(FChildren);
  inherited Destroy;
end;

function TNode.CanAddChild(AChild: TNode): Boolean;
begin
  Result:=Assigned(AChild);
end;

procedure TNode.Add(AChild: TNode);
begin
  Assert(Assigned(AChild), 'Tried to add null child');
  Assert(CanAddChild(AChild), 'Cannot add the given child');
  if not CanAddChild(AChild) then Exit;
  AChild.FParent:=Self;
  FChildren.Add(AChild);
  ChildAdded(AChild);
  AChild.ParentChanged;
  ChildrenChanged;
end;

procedure TNode.Remove(AChild: TNode);
begin
  Assert(Assigned(AChild), 'Tried to remove null child');
  Assert(HasChild(AChild), 'The given child is not a descendant of this node');
  FChildren.Remove(AChild);
  ChildRemoved(AChild);
  AChild.FParent:=nil;
  AChild.ParentChanged;
  ChildrenChanged;
end;

procedure TNode.Delete(AChild: TNode);
begin
  Remove(AChild);
  AChild.Free;
end;

procedure TNode.Clear;
var
  I: Integer;
begin
  if HasChildren then begin
    for I:=ChildCount - 1 downto 0 do begin
      ChildRemoved(Children[I]);
      Children[I].FParent:=nil;
      Children[I].ParentChanged;
      Children[I].Free;
    end;
    FChildren.Clear;
    ChildrenChanged;
  end;
end;

procedure TNode.Swap(AChild1, AChild2: TNode);
begin
  Assert(Assigned(AChild1), 'Tried to swap a null child');
  Assert(Assigned(AChild2), 'Tried to swap a null child');
  Assert(HasChild(AChild1), 'The first child is not a descendant of this node');
  Assert(HasChild(AChild2), 'The second child is not a descendant of this node');
  FChildren.Exchange(FChildren.IndexOf(AChild1), FChildren.IndexOf(AChild2));
  ChildrenChanged;
end;

procedure TNode.Swap(Index1, Index2: Integer);
begin
  Assert((Index1 >= 0) and (Index1 < ChildCount), 'Invalid first index');
  Assert((Index2 >= 0) and (Index2 < ChildCount), 'Invalid second index');
  FChildren.Exchange(Index1, Index2);
  ChildrenChanged;
end;

function TNode.IndexOf(AChild: TNode): Integer;
begin
  Assert(Assigned(AChild), 'Tried to obtain the index of a null child');
  Result:=FChildren.IndexOf(AChild);
end;

function TNode.HasChild(AChild: TNode): Boolean;
begin
  Result:=IndexOf(AChild) >= 0;
end;

function TNode.FindChild(AName: string; Deep: Boolean): TNode;
var
  I: Integer;
begin
  for I:=0 to ChildCount - 1 do begin
    if Children[I].Name=AName then Exit(Children[I]);
    if Deep then begin
      Result:=Children[I].FindChild(AName);
      if Assigned(Result) then Exit;
    end;
  end;
  Result:=nil;
end;

function TNode.HasChildren: Boolean;
begin
  Result:=FChildren.Count > 0;
end;

function TNode.IsEmpty: Boolean;
begin
  Result:=FChildren.Count=0;
end;

function TNode.GetChildByPath(APath: string; Separator: Char): TNode;
var
  SepPos: Integer;
  DirectChild: TNode;
begin
  SepPos:=Pos(Separator, APath);
  if SepPos=0 then Exit(FindChild(APath, False));
  DirectChild:=FindChild(Copy(APath, 1, SepPos - 1), False);
  if not Assigned(DirectChild) then Exit(nil);
  Result:=DirectChild.GetChildByPath(Copy(APath, SepPos + 1, MaxInt), Separator);
end;

function TNode.GetPath(Separator: Char): string;
begin
  if Assigned(Parent) then
    Result:=Parent.GetPath + Separator + Name
  else
    Result:=Name;
end;

function TNode.GetUniqueChildName(Prefix: string; Deep: Boolean): string;
var
  I: Integer;
begin
  for I:=0 to MaxInt do begin
    Result:=Prefix + IntToStr(I);
    if not Assigned(FindChild(Result, Deep)) then Exit;
  end;
  Result:=GetUniqueChildName(Prefix + '_', Deep);
end;

function TNode.CanModify: Boolean;
begin
  if Assigned(Parent) then
    Result:=Parent.CanModify
  else
    Result:=True;
end;

function TNode.MarkModified: Boolean;
begin
  if IsModified or not CanModify then Exit(False);
  if Assigned(Parent) and not Parent.MarkModified then Exit(False);
  FNodeFlags:=FNodeFlags + [nfModified];
  ModifyChanged;
  Result:=True;
end;

procedure TNode.ClearModified;
var
  I: Integer;
begin
  if not IsModified then Exit;
  FNodeFlags:=FNodeFlags - [nfModified];
  for I:=0 to ChildCount - 1 do Children[I].ClearModified;
  ModifyChanged;
end;

function TNode.IsModified: Boolean;
begin
  Result:=nfModified in FNodeFlags;
end;

function TNode.GetRootNode: TNode;
begin
  Result:=Self;
  while Assigned(Result.Parent) do Result:=Result.Parent;
end;

function TNode.GetAncestorWithClass(AClass: TNodeClass): TNode;
begin
  Result:=Self;
  while Assigned(Result) and not (Result is AClass) do Result:=Result.Parent;
end;

function TNode.GetChildrenWithClass(AClass: TNodeClass; Deep: Boolean): TNodeArray;
var
  R: TNodeArray;

  procedure Scan(ANode: TNode);
  var
    I: Integer;
  begin
    for I:=0 to ANode.ChildCount - 1 do begin
      if ANode.Children[I] is AClass then begin
        SetLength(R, Length(R) + 1);
        R[High(R)]:=ANode.Children[I];
      end;
      if Deep then Scan(ANode.Children[I]);
    end;
  end;

begin
  SetLength(R, 0);
  Scan(Self);
  Result:=R;
end;

end.

