unit Transformables;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Nodes, Maths, FGL;

type
  TTransformable = class;
  TTransformableList = specialize TFPGObjectList<TTransformable>;

  { TTransformable }

  TTransformable = class(TNode)
  private
    FTransformableChildren: TTransformableList;
    LocalDirty, WorldDirty: Boolean;
    FLocalMatrix, FWorldMatrix: TMatrix;
    function GetLocalMatrix: TMatrix; inline;
    function GetTransformableParent: TTransformable; inline;
    function GetTransformableChildCount: Integer; inline;
    function GetTransformableChildren(AIndex: Integer): TTransformable; inline;
    function GetWorldMatrix: TMatrix; inline;
    procedure UpdateLocalMatrix;
    procedure UpdateWorldMatrix;
  protected
    procedure InvalidateLocalMatrix;
    procedure CalculateLocalMatrix(out AMatrix: TMatrix); virtual;
    procedure ParentChanged; override;
    procedure ChildAdded(AChild: TNode); override;
    procedure ChildRemoved(AChild: TNode); override;
  public
    destructor Destroy; override;
    property TransformableParent: TTransformable read GetTransformableParent;
    property TransformableChildren[AIndex: Integer]: TTransformable read GetTransformableChildren;
    property TransformableChildCount: Integer read GetTransformableChildCount;
    property LocalMatrix: TMatrix read GetLocalMatrix;
    property WorldMatrix: TMatrix read GetWorldMatrix;
  end;

implementation

{ TTransformable }

function TTransformable.GetTransformableParent: TTransformable;
begin
  Result:=Parent as TTransformable;
end;

function TTransformable.GetTransformableChildCount: Integer;
begin
  Result:=FTransformableChildren.Count;
end;

function TTransformable.GetTransformableChildren(AIndex: Integer): TTransformable;
begin
  Result:=FTransformableChildren[AIndex];
end;

function TTransformable.GetLocalMatrix: TMatrix;
begin
  if LocalDirty then UpdateLocalMatrix;
  Result:=FLocalMatrix;
end;

function TTransformable.GetWorldMatrix: TMatrix;
begin
  if WorldDirty then UpdateWorldMatrix;
  Result:=FWorldMatrix;
end;

procedure TTransformable.UpdateLocalMatrix;
begin
  Assert(LocalDirty, 'Tried to update the local matrix without the localdirty flag');
  CalculateLocalMatrix(FLocalMatrix);
  LocalDirty:=False;
end;

procedure TTransformable.UpdateWorldMatrix;
begin
  Assert(WorldDirty, 'Tried to update the local matrix without the worlddirty flag');
  FWorldMatrix:=LocalMatrix;
  if Assigned(TransformableParent) then FWorldMatrix.Multiply(TransformableParent.WorldMatrix);
  WorldDirty:=False;
end;

procedure TTransformable.InvalidateLocalMatrix;
var
  I: Integer;
begin
  LocalDirty:=False;
  WorldDirty:=False;
  for I:=0 to TransformableChildCount - 1 do
    TransformableChildren[I].WorldDirty:=True;
end;

procedure TTransformable.CalculateLocalMatrix(out AMatrix: TMatrix);
begin
end;

procedure TTransformable.ParentChanged;
begin
  inherited ParentChanged;
  InvalidateLocalMatrix;
end;

procedure TTransformable.ChildAdded(AChild: TNode);
begin
  inherited ChildAdded(AChild);
  if AChild is TTransformable then begin
    if not Assigned(FTransformableChildren) then FTransformableChildren:=TTransformableList.Create(False);
    FTransformableChildren.Add(TTransformable(AChild));
  end;
end;

procedure TTransformable.ChildRemoved(AChild: TNode);
begin
  if AChild is TTransformable then begin
    FTransformableChildren.Remove(TTransformable(AChild));
  end;
  inherited ChildRemoved(AChild);
end;

destructor TTransformable.Destroy;
begin
  FreeAndNil(FTransformableChildren);
  inherited Destroy;
end;

end.

