unit TreeText;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

type

  { TTNode }

  TTNode = class
  private
    FParent: TTNode;
    FChildren: array of TTNode;
    FName: string;
    FValue: string;
    function GetChildCount: Integer; inline;
    function GetChildren(AIndex: Integer): TTNode; inline;
    function GetCode(Padding: Integer): string;
    function GetCode: string; inline;
    function GetValues(AName: string): string; inline;
    procedure SetCode(AValue: string; var Head: Integer);
    procedure SetCode(AValue: string);
    procedure SetName(AValue: string);
    procedure SetValue(AValue: string);
    procedure SetValues(AName: string; AValue: string);
  public
    destructor Destroy; override;
    procedure Clear;
    procedure AddChild(ANode: TTNode);
    procedure RemoveChild(ANode: TTNode);
    function CreateChild(AName: string=''): TTNode;
    function FindNode(ANode: TTNode): Integer;
    function FindChild(AName: string; CreateNew: Boolean=False): TTNode;
    function SetChildValue(AName, AValue: string): TTNode;
    property Name: string read FName write SetName;
    property Value: string read FValue write SetValue;
    property Values[AName: string]: string read GetValues write SetValues; default;
    property Parent: TTNode read FParent;
    property Children[AIndex: Integer]: TTNode read GetChildren;
    property ChildCount: Integer read GetChildCount;
    property Code: string read GetCode write SetCode;
  end;

  { TTreeText }

  TTreeText = class(TComponent)
  private
    FRoot: TTNode;
    function GetCode: string; inline;
    function GetValues(AName: string): string; inline;
    procedure SetCode(AValue: string); inline;
    procedure SetValues(AName: string; AValue: string); inline;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SaveToStream(AStream: TStream);
    procedure SaveToFile(AFilename: TFilename);
    procedure LoadFromStream(AStream: TStream);
    procedure LoadFromFile(AFilename: TFilename);
    property Root: TTNode read FRoot;
    property Values[AName: string]: string read GetValues write SetValues; default;
  published
    property Code: string read GetCode write SetCode;
  end;

procedure Register;

implementation

uses
  LResources, StrUtils;

procedure Register;
begin
  {$I treetext_icon.lrs}
  RegisterComponents('RTTK', [TTreeText]);
end;

function Escaped(Str: string): string;
var
  I: Integer;
  NeedQuotes: Boolean;
begin
  Result:='';
  NeedQuotes:=False;
  for I:=1 to Length(Str) do begin
    if not (Str[I] in ['a'..'z', 'A'..'Z', '0'..'9', '_']) then NeedQuotes:=True;
    if Str[I] in ['"', '\'] then begin
      NeedQuotes:=True;
      Result:=Result + '\' + Str[I];
    end else if Str[I] in ['.', '='] then begin
      NeedQuotes:=True;
      Result:=Result + Str[I];
    end else if (Ord(Str[I]) < 32) or (Ord(Str[I]) > 127) then begin
      NeedQuotes:=True;
      Result:=Result + '\x' + HexStr(Ord(Str[I]), 2);
    end else
      Result:=Result + Str[I];
  end;
  if NeedQuotes then Result:='"' + Result + '"';
end;

{ TTreeText }

function TTreeText.GetCode: string;
begin
  Result:=Root.GetCode;
end;

function TTreeText.GetValues(AName: string): string;
begin
  Result:=Root[AName];
end;

procedure TTreeText.SetCode(AValue: string);
begin
  Root.SetCode(AValue);
end;

procedure TTreeText.SetValues(AName: string; AValue: string);
begin
  Root[AName]:=AValue;
end;

constructor TTreeText.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FRoot:=TTNode.Create;
end;

destructor TTreeText.Destroy;
begin
  FreeAndNil(FRoot);
  inherited Destroy;
end;

procedure TTreeText.SaveToStream(AStream: TStream);
var
  TheCode: RawByteString;
begin
  TheCode:=Root.Code;
  AStream.WriteBuffer((@TheCode[1])^, Length(TheCode));
end;

procedure TTreeText.SaveToFile(AFilename: TFilename);
var
  FS: TFileStream;
begin
  try
    FS:=TFileStream.Create(AFilename, fmCreate);
    SaveToStream(FS);
  finally
    FreeAndNil(FS);
  end;
end;

procedure TTreeText.LoadFromStream(AStream: TStream);
var
  Buffer: array [1..65536] of Byte;
  I, BytesRead: LongInt;
  TheCode: RawByteString;
begin
  Root.Clear;
  TheCode:='';
  while True do begin
    BytesRead:=AStream.Read(Buffer{%H-}, SizeOf(Buffer));
    if BytesRead=0 then Break;
    for I:=1 to BytesRead do TheCode:=TheCode + Chr(Buffer[I]);
  end;
  Root.SetCode(TheCode);
end;

procedure TTreeText.LoadFromFile(AFilename: TFilename);
var
  FS: TFileStream;
begin
  try
    FS:=TFileStream.Create(AFilename, fmOpenRead);
    LoadFromStream(FS);
  finally
    FreeAndNil(FS);
  end;
end;

{ TTNode }

function TTNode.GetChildCount: Integer;
begin
  Result:=Length(FChildren);
end;

function TTNode.GetChildren(AIndex: Integer): TTNode;
begin
  Result:=FChildren[AIndex];
end;

function TTNode.GetCode(Padding: Integer): string;
var
  I: Integer;
  ChildCode: String;

  function GetPadding: string;
  var
    I: Integer;
  begin
    Result:='';
    for I:=1 to Padding do Result:=Result + '  ';
  end;

begin
  Result:=GetPadding + Escaped(Name);
  if Value <> '' then Result:=Result + '=' + Escaped(Value);
  if ChildCount=0 then Exit;
  if (Name <> '') or (Value <> '') then Result:=Result + ' ';
  if not ((Name='') and (Parent=nil)) then Result:=Result + '{' + LineEnding;
  for I:=0 to High(FChildren) do begin
    if (Children[I].Value='') and (Children[I].ChildCount=0) then Continue;
    ChildCode:=Children[I].GetCode(Padding + 1);
    Result:=Result + ChildCode;
    if Pos(LineEnding, ChildCode)=0 then
      Result:=Result + LineEnding;
  end;
  if not ((Name='') and (Parent=nil)) then Result:=Result + GetPadding + '}' + LineEnding;
end;

function TTNode.GetCode: string;
begin
  if Assigned(Parent) then Result:=GetCode(0) else Result:=GetCode(-1);
end;

function TTNode.GetValues(AName: string): string;
begin
  Result:=FindChild(AName, True).Value;
end;

procedure TTNode.SetCode(AValue: string; var Head: Integer);

  procedure SkipSpaces;
  begin
    while (Head <= Length(AValue)) and (AValue[Head] in [#9, #10, #13, ' ', '#']) do begin
      if AValue[Head]='#' then begin
        while (Head <= Length(AValue)) and not (AValue[Head] in [#10, #13]) do Inc(Head);
        Continue;
      end;
      Inc(Head);
    end;
  end;

  function Scan: string;
  begin
    Result:='';
    SkipSpaces;
    if Head=Length(AValue) then Exit;
    if AValue[Head]='"' then begin
      Inc(Head);
      while Head <= Length(AValue) do begin
        if (AValue[Head]='\') and (Head < Length(AValue)) then begin
          case AValue[Head + 1] of
            'x': if Head + 3 <= Length(AValue) then begin
              Result:=Result + Chr(Hex2Dec(AValue[Head + 2] + AValue[Head + 3]));
              Inc(Head, 4);
              Continue;
            end;
            else begin
              Result:=Result + AValue[Head + 1];
              Inc(Head, 2);
              Continue;
            end;
          end;
        end;
        if AValue[Head]='"' then begin
          Inc(Head);
          Break;
        end;
        Result:=Result + AValue[Head];
        Inc(Head);
      end;
    end else begin
      while Head <= Length(AValue) do begin
        if AValue[Head] in [#9, #10, #13, ' ', '=', '{', '}'] then Break;
        Result:=Result + AValue[Head];
        Inc(Head);
      end;
    end;
  end;

var
  Child: TTNode;
  SaveHead: Integer;
begin
  Clear;
  SkipSpaces;
  if Head >= Length(AValue) then Exit;
  FName:=Scan;
  SkipSpaces;
  if Head >= Length(AValue) then Exit;
  if AValue[Head]='=' then begin
    Inc(Head);
    FValue:=Scan;
  end;
  SkipSpaces;
  if Head >= Length(AValue) then Exit;
  if AValue[Head]='{' then begin
    Inc(Head);
    while Head <= Length(AValue) do begin
      SkipSpaces;
      if Head >= Length(AValue) then Exit;
      if AValue[Head]='}' then begin
        Inc(Head);
        Break;
      end;
      SaveHead:=Head;
      Child:=TTNode.Create;
      Child.FParent:=Self;
      Child.SetCode(AValue, Head);
      if Trim(Child.Name)='' then begin
        Child.Free;
      end else begin
        SetLength(FChildren, Length(FChildren) + 1);
        FChildren[High(FChildren)]:=Child;
      end;
      if Head=SaveHead then Break;
    end;
  end;
end;

procedure TTNode.SetCode(AValue: string);
var
  Head: Integer;
begin
  Clear;
  Head:=1;
  AValue:=Trim(AValue);
  if AValue='' then Exit;
  if Assigned(Parent) then SetCode(AValue, Head) else SetCode('{' + AValue + '}', Head);
end;

procedure TTNode.SetName(AValue: string);
begin
  if FName=AValue then Exit;
  FName:=AValue;
end;

procedure TTNode.SetValue(AValue: string);
begin
  if FValue=AValue then Exit;
  FValue:=AValue;
end;

procedure TTNode.SetValues(AName: string; AValue: string);
begin
  FindChild(AName, True).Value:=AValue;
end;

destructor TTNode.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TTNode.Clear;
var
  I: Integer;
begin
  FName:='';
  FValue:='';
  for I:=0 to High(FChildren) do FChildren[I].Free;
  SetLength(FChildren, 0);
end;

procedure TTNode.AddChild(ANode: TTNode);
begin
  if Assigned(ANode.Parent) then ANode.Parent.RemoveChild(ANode);
  ANode.FParent:=Self;
  SetLength(FChildren, Length(FChildren) + 1);
  FChildren[High(FChildren)]:=ANode;
end;

procedure TTNode.RemoveChild(ANode: TTNode);
var
  I, Index: Integer;
begin
  Assert(ANode.Parent=Self, 'The given node is not a child of this node');
  ANode.FParent:=nil;
  Index:=FindNode(ANode);
  for I:=Index to High(FChildren) - 1 do FChildren[I]:=FChildren[I + 1];
  SetLength(FChildren, Length(FChildren) - 1);
end;

function TTNode.CreateChild(AName: string): TTNode;
begin
  Result:=TTNode.Create;
  Result.Name:=AName;
  AddChild(Result);
end;

function TTNode.FindNode(ANode: TTNode): Integer;
var
  I: Integer;
begin
  for I:=0 to High(FChildren) do
    if FChildren[I]=ANode then Exit(i);
  Result:=-1;
end;

function TTNode.FindChild(AName: string; CreateNew: Boolean): TTNode;
var
  Node: TTNode;

  function FindDirectChild(ADirectName: string): TTnode;
  var
    I: Integer;
  begin
    Result:=nil;
    for I:=0 to ChildCount - 1 do begin
      if Children[I].Name=ADirectName then begin
        Result:=Children[I];
        Break;
      end;
    end;
    if (Result=nil) and CreateNew then begin
      Result:=TTNode.Create;
      Result.FParent:=Self;
      Result.Name:=ADirectName;
      SetLength(FChildren, Length(FChildren) + 1);
      FChildren[High(FChildren)]:=Result;
    end;
  end;

begin
  if Pos('.', AName) <> 0 then begin
    Node:=FindDirectChild(Copy(AName, 1, Pos('.', AName) - 1));
    if Assigned(Node) then
      Result:=Node.FindChild(Copy(AName, Pos('.', AName) + 1, MaxInt), CreateNew)
    else
      Result:=nil;
  end else
    Result:=FindDirectChild(AName);
end;

function TTNode.SetChildValue(AName, AValue: string): TTNode;
begin
  Result:=FindChild(AName);
  Result.Value:=AValue;
end;

end.

