unit Widgets3d;

{$mode objfpc}{$H+}
{$interfaces corba}

interface

uses
  Classes, SysUtils, Controls, FGL, Maths, Textures;

type
  T3DWidget = class;
  T3DWidgetManager = class;
  T3DWidgetList = specialize TFPGObjectList<T3DWidget>;
  T3DTransformationWidgetOperation = (twoTranslation, twoRotation, twoScale);
  T3DTransformationWidgetOperations = set of T3DTransformationWidgetOperation;
  T3DObjectTransformationWidgetCoordinateSystem = (tcsLocal, tcsGlobal, tcsCustom);

  T3DWidgetMouseButtonEvent = procedure(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState) of object;
  T3DWidgetMouseMotionEvent = procedure(Sender: TObject; const ARay: TRay; const APoint: TVector) of object;

  T3DTransformationWidgetBeginTransformEvent = procedure(Sender: TObject; const ATransform: TTransform; var Veto: Boolean; AShift: TShiftState) of object;
  T3DTransformationWidgetEndTransformEvent = procedure(Sender: TObject; const ATransform: TTransform; AShift: TShiftState) of object;
  T3DTransformationWidgetTransformEvent = procedure(Sender: TObject; const OldTransform: TTransform; var NewTransform: TTransform; var Veto: Boolean) of object;

const
  Default3DTransformationWidgetOperations = [twoTranslation, twoRotation, twoScale];

type

  { T3DWidget }

  T3DWidget = class(TComponent)
  private
    FHoverColor: TExtColor;
    FNormalColor: TExtColor;
    FOnActivated: TNotifyEvent;
    FOnDeactivated: TNotifyEvent;
    FOnMouseDown: T3DWidgetMouseButtonEvent;
    FOnMouseEnter: TNotifyEvent;
    FOnMouseLeave: TNotifyEvent;
    FOnMouseMotion: T3DWidgetMouseMotionEvent;
    FOnMouseUp: T3DWidgetMouseButtonEvent;
    FWidgetManager: T3DWidgetManager;
  protected
    procedure MouseDown(const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState); virtual;
    procedure MouseUp(const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState); virtual;
    procedure MouseMotion(const ARay: TRay; const APoint: TVector); virtual;
    procedure MouseEnter; virtual;
    procedure MouseLeave; virtual;
    procedure AddedToManager; virtual;
    procedure RemovedFromManager; virtual;
    procedure Activated; virtual;
    procedure Deactivated; virtual;
    procedure Render; virtual; abstract;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function RayHit(const ARay: TRay; out IP: TVector): Boolean; virtual; abstract;
    function IsMouseWidget: Boolean;
    function Color: TExtColor; inline;
    property NormalColor: TExtColor read FNormalColor write FNormalColor;
    property HoverColor: TExtColor read FHoverColor write FHoverColor;
    property WidgetManager: T3DWidgetManager read FWidgetManager;
  published
    property OnMouseDown: T3DWidgetMouseButtonEvent read FOnMouseDown write FOnMouseDown;
    property OnMouseUp: T3DWidgetMouseButtonEvent read FOnMouseUp write FOnMouseUp;
    property OnMouseMotion: T3DWidgetMouseMotionEvent read FOnMouseMotion write FOnMouseMotion;
    property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
    property OnActivated: TNotifyEvent read FOnActivated write FOnActivated;
    property OnDeactivated: TNotifyEvent read FOnDeactivated write FOnDeactivated;
  end;

  { T3DScaledWidget }

  T3DScaledWidget = class(T3DWidget)
  private
    FPosition: TVector;
    procedure SetPosition(AValue: TVector);
  protected
    FPixelScale: Double;
    procedure UpdatePixelScale;
  public
    property Position: TVector read FPosition write SetPosition;
    property PixelScale: Double read FPixelScale;
  end;

  { T3DSphericalWidget }

  T3DSphericalWidget = class(T3DScaledWidget)
  private
    FHoverRadius: Double;
    FNormalRadius: Double;
  protected
    procedure Render; override;
  public
    constructor Create(AOwner: TComponent); override;
    function RayHit(const ARay: TRay; out IP: TVector): Boolean; override;
    property NormalRadius: Double read FNormalRadius write FNormalRadius;
    property HoverRadius: Double read FHoverRadius write FHoverRadius;
  end;

  T3DDirectionalWidgetStyle = (dwsArrow, dwsBox);

  { T3DDirectionalWidget }

  T3DDirectionalWidget = class(T3DScaledWidget)
  private
    FNormal: TVector;
    FStyle: T3DDirectionalWidgetStyle;
  protected
    procedure Render; override;
  public
    constructor Create(AOwner: TComponent); override;
    function RayHit(const ARay: TRay; out IP: TVector): Boolean; override;
    property Normal: TVector read FNormal write FNormal;
    property Style: T3DDirectionalWidgetStyle read FStyle write FStyle;
  end;

  { T3DGeometricWidget }

  T3DGeometricWidget = class(T3DScaledWidget)
  private
    FVertices: array of Single;
    FScaledVertices: array of Single;
    FCalcScaled: Boolean;
    procedure CalcScaled;
  protected
    procedure Render; override;
  public
    constructor Create(AOwner: TComponent); override;
    function RayHit(const ARay: TRay; out IP: TVector): Boolean; override;
    procedure ClearGeometry;
    procedure AddFace(const A, B, C: TVector);
    procedure AddTriangles(const AVertices: array of Single; const Matrix: TMatrix);
  end;

  { T3DCompoundWidget }

  T3DCompoundWidget = class(T3DWidget)
  private
    FWidgets: T3DWidgetList;
    function GetWidgetCount: Integer; inline;
    function GetWidgets(AIndex: Integer): T3DWidget; inline;
  protected
    procedure AddedToManager; override;
    procedure RemovedFromManager; override;
    procedure Render; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function RayHit(const ARay: TRay; out IP: TVector): Boolean; override;
    procedure Clear;
    procedure Add(AWidget: T3DWidget);
    procedure Remove(AWidget: T3DWidget);
    procedure Delete(AWidget: T3DWidget);
    function IndexOf(AWidget: T3DWidget): Integer; inline;
    property Widgets[AIndex: Integer]: T3DWidget read GetWidgets;
    property WidgetCount: Integer read GetWidgetCount;
  end;

  { T3DTransformationWidget }

  T3DTransformationWidget = class(T3DCompoundWidget)
  private
    FAxisMask: TAxes;
    FOnChange: TNotifyEvent;
    FOnTransform: T3DTransformationWidgetTransformEvent;
    FOperations: T3DTransformationWidgetOperations;
    FTransform: TTransform;
    FOnBeginTransform: T3DTransformationWidgetBeginTransformEvent;
    FOnEndTransform: T3DTransformationWidgetEndTransformEvent;
    procedure SetAxisMask(AValue: TAxes);
    procedure SetOperations(AValue: T3DTransformationWidgetOperations);
    procedure SetTransform(AValue: TTransform);
  protected
    procedure TransformChanged; virtual;
    procedure OperationsChanged; virtual;
    procedure AxisMaskChanged; virtual;
    function BeginTransform(AShift: TShiftState): Boolean;
    procedure EndTransform(AShift: TShiftState);
    procedure ApplyTransform(ATransform: TTransform);
  public
    constructor Create(AOwner: TComponent); override;
    property Transform: TTransform read FTransform write SetTransform;
  published
    property Operations: T3DTransformationWidgetOperations read FOperations write SetOperations default Default3DTransformationWidgetOperations;
    property AxisMask: TAxes read FAxisMask write SetAxisMask default AllAxes;
    property OnBeginTransform: T3DTransformationWidgetBeginTransformEvent read FOnBeginTransform write FOnBeginTransform;
    property OnEndTransform: T3DTransformationWidgetEndTransformEvent read FOnEndTransform write FOnEndTransform;
    property OnTransform: T3DTransformationWidgetTransformEvent read FOnTransform write FOnTransform;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  { T3DObjectTransformationWidget }

  T3DObjectTransformationWidget = class(T3DTransformationWidget)
  private
    FCoordinateSystem: T3DObjectTransformationWidgetCoordinateSystem;
    FFlipTowardsCamera: Boolean;
    FForceUniformScale: Boolean;
    FOriginAxisMask: TAxes;
    FRotationAxes: TAxes;
    FScaleAxes: TAxes;
    FShowOrigin: Boolean;
    FTranslationAxes: TAxes;
    Origin: T3DSphericalWidget;
    XAxis, YAxis, ZAxis: T3DDirectionalWidget;
    XYPlane, XZPlane, ZYPlane: T3DGeometricWidget;
    XRot, YRot, ZRot: T3DGeometricWidget;
    XScale, YScale, ZScale: T3DGeometricWidget;
    OriginalTransform: TTransform;
    ActionOffset: TVector;
    MovingAxis: T3DDirectionalWidget;
    MovingPlane: Boolean;
    MotionPlane: TPlane;
    Rotating: Boolean;
    Scaling, Dual, Uniform: Boolean;
    CustomXAxis, CustomYAxis, CustomZAxis: TVector;
    FlippedAxes: TAxes;
    procedure AxisMouseDown(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure AxisMouseUp(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure AxisMouseMotion(Sender: TObject; const ARay: TRay; const APoint: TVector);
    procedure PlaneMouseDown(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure PlaneMouseUp(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure PlaneMouseMotion(Sender: TObject; const ARay: TRay; const APoint: TVector);
    procedure RotationMouseDown(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure RotationMouseUp(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure RotationMouseMotion(Sender: TObject; const ARay: TRay; const APoint: TVector);
    procedure ScaleMouseDown(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure ScaleMouseUp(Sender: TObject; const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
    procedure ScaleMouseMotion(Sender: TObject; const ARay: TRay; const APoint: TVector);
    procedure SetCoordinateSystem(AValue: T3DObjectTransformationWidgetCoordinateSystem);
    procedure SetFlipTowardsCamera(AValue: Boolean);
    procedure SetRotationAxes(AValue: TAxes);
    procedure SetScaleAxes(AValue: TAxes);
    procedure SetShowOrigin(AValue: Boolean);
    procedure SetTranslationAxes(AValue: TAxes);
    procedure UpdateFromTransform;
    procedure UpdateAxisFlip;
    procedure UpdateWidgets;
  protected
    procedure AddedToManager; override;
    procedure TransformChanged; override;
    procedure OperationsChanged; override;
    procedure AxisMaskChanged; override;
    procedure Render; override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure SetCustomCoordinateSystem(const AXAxis, AYAxis, AZAxis: TVector);
    procedure SetCustomCoordinateSystem(const ATransform: TTransform);
  published
    property FlipTowardsCamera: Boolean read FFlipTowardsCamera write SetFlipTowardsCamera default True;
    property ShowOrigin: Boolean read FShowOrigin write SetShowOrigin default True;
    property TranslationAxes: TAxes read FTranslationAxes write SetTranslationAxes default AllAxes;
    property RotationAxes: TAxes read FRotationAxes write SetRotationAxes default AllAxes;
    property ScaleAxes: TAxes read FScaleAxes write SetScaleAxes default AllAxes;
    property ForceUniformScale: Boolean read FForceUniformScale write FForceUniformScale default False;
    property OriginAxisMask: TAxes read FOriginAxisMask write FOriginAxisMask default AllAxes;
    property CoordinateSystem: T3DObjectTransformationWidgetCoordinateSystem read FCoordinateSystem write SetCoordinateSystem default tcsLocal;
  end;

  { I3DWidgetManagerEnvironment }

  I3DWidgetManagerEnvironment = interface
    function GetForwardDirection: TVector;
    function GetUpDirection: TVector;
    function GetRightDirection: TVector;
    function GetPixelScale(const APosition: TVector): Double;
    function IsActiveEnvironment: Boolean;
  end;

  { T3DWidgetManager }

  T3DWidgetManager = class(TComponent)
  private
    FActiveWidget: T3DWidget;
    FEnvironment: I3DWidgetManagerEnvironment;
    FWidgets: T3DWidgetList;
    FMouseWidget: T3DWidget;
    FHitPlane: TPlane;
    function GetWidgetCount: Integer; inline;
    function GetWidgets(AIndex: Integer): T3DWidget; inline;
    procedure SetActiveWidget(AValue: T3DWidget);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function FindWidgetByRay(const ARay: TRay; out IP: TVector): T3DWidget;
    procedure MouseDown(const ARay: TRay; AButton: TMouseButton; AShift: TShiftState);
    procedure MouseUp(const ARay: TRay; AButton: TMouseButton; AShift: TShiftState);
    procedure MouseMotion(const ARay: TRay);
    procedure Render;
    procedure Clear;
    procedure Add(AWidget: T3DWidget);
    procedure Remove(AWidget: T3DWidget);
    procedure Delete(AWidget: T3DWidget);
    function IndexOf(AWidget: T3DWidget): Integer; inline;
    function Has(AWidget: T3DWidget): Boolean; inline;
    property Widgets[AIndex: Integer]: T3DWidget read GetWidgets;
    property WidgetCount: Integer read GetWidgetCount;
    property ActiveWidget: T3DWidget read FActiveWidget write SetActiveWidget;
    property MouseWidget: T3DWidget read FMouseWidget;
    property HitPlane: TPlane read FHitPlane;
    property Environment: I3DWidgetManagerEnvironment read FEnvironment write FEnvironment;
  end;

procedure Register;

implementation

uses
  GL, LResources, UtilMeshData;

procedure Register;
begin
  {$I widgets3d_icon.lrs}
  RegisterComponents('RTTK', [T3DWidgetManager, T3DObjectTransformationWidget]);
end;

{ T3DTransformationWidget }

procedure T3DTransformationWidget.SetTransform(AValue: TTransform);
begin
  if FTransform=AValue then Exit;
  FTransform:=AValue;
  TransformChanged;
  if Assigned(OnChange) then OnChange(Self);
end;

procedure T3DTransformationWidget.SetOperations(
  AValue: T3DTransformationWidgetOperations);
begin
  if FOperations=AValue then Exit;
  FOperations:=AValue;
  OperationsChanged;
end;

procedure T3DTransformationWidget.SetAxisMask(AValue: TAxes);
begin
  if FAxisMask=AValue then Exit;
  FAxisMask:=AValue;
  AxisMaskChanged;
end;

procedure T3DTransformationWidget.TransformChanged;
begin
end;

procedure T3DTransformationWidget.OperationsChanged;
begin
end;

procedure T3DTransformationWidget.AxisMaskChanged;
begin
end;

function T3DTransformationWidget.BeginTransform(AShift: TShiftState): Boolean;
begin
  Result:=False;
  if Assigned(FOnBeginTransform) then
    FOnBeginTransform(Self, Transform, Result, AShift);
  Result:=not Result;
end;

procedure T3DTransformationWidget.EndTransform(AShift: TShiftState);
begin
  if Assigned(FOnEndTransform) then
    FOnEndTransform(Self, Transform, AShift)
end;

procedure T3DTransformationWidget.ApplyTransform(ATransform: TTransform);
var
  Veto: Boolean;
begin
  if Assigned(OnTransform) then begin
    Veto:=False;
    OnTransform(Self, Transform, ATransform, Veto);
    if Veto then Exit;
  end;
  if not (twoTranslation in Operations) then ATransform.Translation:=Transform.Translation;
  if not (twoRotation in Operations) then ATransform.Rotation:=Transform.Rotation;
  if not (twoScale in Operations) then ATransform.Scale:=Transform.Scale;
  if not (axX in AxisMask) then begin
    ATransform.Translation.x:=Transform.Translation.x;
    ATransform.Rotation.x:=Transform.Rotation.x;
    ATransform.Scale.x:=Transform.Scale.x;
  end;
  if not (axY in AxisMask) then begin
    ATransform.Translation.y:=Transform.Translation.y;
    ATransform.Rotation.y:=Transform.Rotation.y;
    ATransform.Scale.y:=Transform.Scale.y;
  end;
  if not (axZ in AxisMask) then begin
    ATransform.Translation.z:=Transform.Translation.z;
    ATransform.Rotation.z:=Transform.Rotation.z;
    ATransform.Scale.z:=Transform.Scale.z;
  end;
  Transform:=ATransform;
end;

constructor T3DTransformationWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTransform.Reset;
  FOperations:=Default3DTransformationWidgetOperations;
end;

{ T3DGeometricWidget }

constructor T3DGeometricWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

procedure T3DGeometricWidget.CalcScaled;
var
  I: Integer;
  Center: TVector;
begin
  I:=0;
  Center.Zero;
  SetLength(FScaledVertices, Length(FVertices));
  if Length(FVertices) > 2 then begin
    while I < Length(FVertices) do begin
      Center.x += FVertices[I];
      Center.y += FVertices[I + 1];
      Center.z += FVertices[I + 2];
      Inc(I, 3);
    end;
    Center.Scale(1/(Length(FVertices)/3));
    I:=0;
    while I < Length(FVertices) do begin
      FScaledVertices[I]:=(FVertices[I] - Center.x)*1.1 + Center.x;
      FScaledVertices[I + 1]:=(FVertices[I + 1] - Center.y)*1.1 + Center.y;
      FScaledVertices[I + 2]:=(FVertices[I + 2] - Center.z)*1.1 + Center.z;
      Inc(I, 3);
    end;
  end;
  FCalcScaled:=False;
end;

procedure T3DGeometricWidget.Render;
begin
  if Length(FVertices) > 0 then begin
    UpdatePixelScale;
    if FCalcScaled then CalcScaled;
    glPushMatrix();
    glTranslated(Position.x, Position.y, Position.z);
    glScaled(PixelScale, PixelScale, PixelScale);
    if IsMouseWidget then begin
      DrawOpenGLTrianglesFor(FScaledVertices, IdentityMatrix, HoverColor);
    end else begin
      DrawOpenGLTrianglesFor(FVertices, IdentityMatrix, NormalColor);
    end;
    glPopMatrix();
  end;
end;

function T3DGeometricWidget.RayHit(const ARay: TRay; out IP: TVector): Boolean;
var
  V1, V2, V3: TVector;
  I: Integer;
begin
  UpdatePixelScale;
  if FCalcScaled then CalcScaled;
  I:=0;
  while I < Length(FVertices) do begin
    V1:=Vector(FScaledVertices[I]*PixelScale, FScaledVertices[I + 1]*PixelScale, FScaledVertices[I + 2]*PixelScale);
    V2:=Vector(FScaledVertices[I + 3]*PixelScale, FScaledVertices[I + 4]*PixelScale, FScaledVertices[I + 5]*PixelScale);
    V3:=Vector(FScaledVertices[I + 6]*PixelScale, FScaledVertices[I + 7]*PixelScale, FScaledVertices[I + 8]*PixelScale);
    V1.Add(Position);
    V2.Add(Position);
    V3.Add(Position);
    if ARay.TriangleHit(V1, V2, V3, IP) then Exit(True);
    Inc(I, 9);
  end;
  Result:=False;
end;

procedure T3DGeometricWidget.ClearGeometry;
begin
  SetLength(FVertices, 0);
end;

procedure T3DGeometricWidget.AddFace(const A, B, C: TVector);
var
  I: Integer;
begin
  I:=Length(FVertices);
  SetLength(FVertices, Length(FVertices) + 1);
  FVertices[I]:=A.x; Inc(I);
  FVertices[I]:=A.y; Inc(I);
  FVertices[I]:=A.z; Inc(I);
  FVertices[I]:=B.x; Inc(I);
  FVertices[I]:=B.y; Inc(I);
  FVertices[I]:=B.z; Inc(I);
  FVertices[I]:=C.x; Inc(I);
  FVertices[I]:=C.y; Inc(I);
  FVertices[I]:=C.z;
  FCalcScaled:=True;
end;

procedure T3DGeometricWidget.AddTriangles(const AVertices: array of Single;
  const Matrix: TMatrix);
var
  V: TVector;
  I, J: Integer;
begin
  Assert((Length(AVertices) mod 3)=0, 'Invalid vertex count');
  I:=Length(FVertices);
  J:=0;
  SetLength(FVertices, Length(FVertices) + Length(AVertices));
  while J < Length(AVertices) do begin
    V.x:=AVertices[J];
    V.y:=AVertices[J + 1];
    V.z:=AVertices[J + 2];
    Matrix.Transform(V);
    FVertices[I]:=V.x;
    FVertices[I + 1]:=V.y;
    FVertices[I + 2]:=V.z;
    Inc(I, 3);
    Inc(J, 3);
  end;
  FCalcScaled:=True;
end;

{ T3DObjectTransformationWidget }

constructor T3DObjectTransformationWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFlipTowardsCamera:=True;
  FShowOrigin:=True;
  FForceUniformScale:=False;
  FAxisMask:=AllAxes;
  FTranslationAxes:=AllAxes;
  FRotationAxes:=AllAxes;
  FScaleAxes:=AllAxes;
  FOriginAxisMask:=AllAxes;
  FCoordinateSystem:=tcsLocal;
  CustomXAxis:=Vector(1, 0, 0);
  CustomYAxis:=Vector(0, 1, 0);
  CustomZAxis:=Vector(0, 0, 1);
  Origin:=T3DSphericalWidget.Create(Self);
  Origin.OnMouseDown:=@PlaneMouseDown;
  Origin.OnMouseUp:=@PlaneMouseUp;
  Origin.OnMouseMotion:=@PlaneMouseMotion;
  XAxis:=T3DDirectionalWidget.Create(Self);
  XAxis.NormalColor:=ExtColor(1, 0, 0);
  XAxis.OnMouseDown:=@AxisMouseDown;
  XAxis.OnMouseUp:=@AxisMouseUp;
  XAxis.OnMouseMotion:=@AxisMouseMotion;
  YAxis:=T3DDirectionalWidget.Create(Self);
  YAxis.NormalColor:=ExtColor(0, 1, 0);
  YAxis.OnMouseDown:=@AxisMouseDown;
  YAxis.OnMouseUp:=@AxisMouseUp;
  YAxis.OnMouseMotion:=@AxisMouseMotion;
  ZAxis:=T3DDirectionalWidget.Create(Self);
  ZAxis.NormalColor:=ExtColor(0, 0, 1);
  ZAxis.OnMouseDown:=@AxisMouseDown;
  ZAxis.OnMouseUp:=@AxisMouseUp;
  ZAxis.OnMouseMotion:=@AxisMouseMotion;
  XYPlane:=T3DGeometricWidget.Create(Self);
  XYPlane.NormalColor:=ExtColor(1, 1, 0);
  XYPlane.OnMouseDown:=@PlaneMouseDown;
  XYPlane.OnMouseUp:=@PlaneMouseUp;
  XYPlane.OnMouseMotion:=@PlaneMouseMotion;
  XZPlane:=T3DGeometricWidget.Create(Self);
  XZPlane.NormalColor:=ExtColor(1, 0, 1);
  XZPlane.OnMouseDown:=@PlaneMouseDown;
  XZPlane.OnMouseUp:=@PlaneMouseUp;
  XZPlane.OnMouseMotion:=@PlaneMouseMotion;
  ZYPlane:=T3DGeometricWidget.Create(Self);
  ZYPlane.NormalColor:=ExtColor(0, 1, 1);
  ZYPlane.OnMouseDown:=@PlaneMouseDown;
  ZYPlane.OnMouseUp:=@PlaneMouseUp;
  ZYPlane.OnMouseMotion:=@PlaneMouseMotion;
  XRot:=T3DGeometricWidget.Create(Self);
  XRot.NormalColor:=ExtColor(0.75, 0.25, 0.25);
  XRot.OnMouseDown:=@RotationMouseDown;
  XRot.OnMouseUp:=@RotationMouseUp;
  XRot.OnMouseMotion:=@RotationMouseMotion;
  YRot:=T3DGeometricWidget.Create(Self);
  YRot.NormalColor:=ExtColor(0.25, 0.75, 0.25);
  YRot.OnMouseDown:=@RotationMouseDown;
  YRot.OnMouseUp:=@RotationMouseUp;
  YRot.OnMouseMotion:=@RotationMouseMotion;
  ZRot:=T3DGeometricWidget.Create(Self);
  ZRot.NormalColor:=ExtColor(0.25, 0.25, 0.75);
  ZRot.OnMouseDown:=@RotationMouseDown;
  ZRot.OnMouseUp:=@RotationMouseUp;
  ZRot.OnMouseMotion:=@RotationMouseMotion;
  XScale:=T3DGeometricWidget.Create(Self);
  XScale.NormalColor:=ExtColor(0.65, 0.35, 0.35);
  XScale.OnMouseDown:=@ScaleMouseDown;
  XScale.OnMouseUp:=@ScaleMouseUp;
  XScale.OnMouseMotion:=@ScaleMouseMotion;
  YScale:=T3DGeometricWidget.Create(Self);
  YScale.NormalColor:=ExtColor(0.35, 0.65, 0.35);
  YScale.OnMouseDown:=@ScaleMouseDown;
  YScale.OnMouseUp:=@ScaleMouseUp;
  YScale.OnMouseMotion:=@ScaleMouseMotion;
  ZScale:=T3DGeometricWidget.Create(Self);
  ZScale.NormalColor:=ExtColor(0.35, 0.35, 0.65);
  ZScale.OnMouseDown:=@ScaleMouseDown;
  ZScale.OnMouseUp:=@ScaleMouseUp;
  ZScale.OnMouseMotion:=@ScaleMouseMotion;
  FTransform.Reset;
  UpdateWidgets;
  UpdateFromTransform;
end;

procedure T3DObjectTransformationWidget.SetCustomCoordinateSystem(const AXAxis, AYAxis, AZAxis: TVector);
begin
  CustomXAxis:=AXAxis.Normalized;
  CustomYAxis:=AYAxis.Normalized;
  CustomZAxis:=AZAxis.Normalized;
  if CoordinateSystem=tcsCustom then UpdateFromTransform;
end;

procedure T3DObjectTransformationWidget.SetCustomCoordinateSystem(const ATransform: TTransform);
var
  Matrix: TMatrix;
begin
  Matrix:=ATransform.ToMatrix;
  SetCustomCoordinateSystem(Matrix.XAxis, Matrix.YAxis, Matrix.ZAxis);
end;

procedure T3DObjectTransformationWidget.AxisMouseDown(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  if not BeginTransform(AShift) then Exit;
  OriginalTransform:=Transform;
  MovingAxis:=T3DDirectionalWidget(Sender);
  DistanceBetweenSegments(
    ARay.o, ARay.o.Added(ARay.d.Scaled(163840)),
    Transform.Translation.Subbed(MovingAxis.Normal.Scaled(163840)),
      Transform.Translation.Added(MovingAxis.Normal.Scaled(163840)),
    @ActionOffset, nil);
  ActionOffset:=ProjectVectorOnVector(ActionOffset, MovingAxis.Normal);
  WidgetManager.ActiveWidget:=MovingAxis;
end;

procedure T3DObjectTransformationWidget.AxisMouseUp(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  MovingAxis:=nil;
  WidgetManager.ActiveWidget:=nil;
  EndTransform(AShift);
end;

procedure T3DObjectTransformationWidget.AxisMouseMotion(Sender: TObject;
  const ARay: TRay; const APoint: TVector);
var
  Delta, IP: TVector;
  NewTransform: TTransform;
begin
  if Assigned(MovingAxis) and not Scaling then begin
    DistanceBetweenSegments(
      ARay.o, ARay.o.Added(ARay.d.Scaled(163840)),
      Transform.Translation.Subbed(MovingAxis.Normal.Scaled(163840)),
        Transform.Translation.Added(MovingAxis.Normal.Scaled(163840)),
      @IP, nil);
    IP:=ProjectVectorOnVector(IP, MovingAxis.Normal);
    Delta:=IP.Subbed(ActionOffset);
    NewTransform:=OriginalTransform;
    NewTransform.Translation:=NewTransform.Translation.Added(Delta);
    ApplyTransform(NewTransform);
  end;
end;

procedure T3DObjectTransformationWidget.PlaneMouseDown(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  if not BeginTransform(AShift) then Exit;
  if Sender=Origin then
    MotionPlane:=WidgetManager.HitPlane
  else if Sender=XYPlane then
    MotionPlane:=Plane(ZAxis.Normal, 0)
  else if Sender=XZPlane then
    MotionPlane:=Plane(YAxis.Normal, 0)
  else if Sender=ZYPlane then
    MotionPlane:=Plane(XAxis.Normal, 0)
  else Exit;
  if not ARay.PlaneHit(MotionPlane, ActionOffset) then Exit;
  MovingPlane:=True;
  OriginalTransform:=Transform;
  WidgetManager.ActiveWidget:=T3DWidget(Sender);
end;

procedure T3DObjectTransformationWidget.PlaneMouseUp(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  MovingPlane:=False;
  WidgetManager.ActiveWidget:=nil;
  EndTransform(AShift);
end;

procedure T3DObjectTransformationWidget.PlaneMouseMotion(Sender: TObject;
  const ARay: TRay; const APoint: TVector);
var
  IP: TVector;
  NewTransform: TTransform;
begin
  if MovingPlane then begin
    if not ARay.PlaneHit(MotionPlane, IP) then Exit;
    NewTransform:=OriginalTransform;
    NewTransform.Translation:=NewTransform.Translation.Added(IP.Subbed(ActionOffset));
    if Sender=Origin then begin
      if not (axX in OriginAxisMask) then NewTransform.Translation.x:=OriginalTransform.Translation.x;
      if not (axY in OriginAxisMask) then NewTransform.Translation.y:=OriginalTransform.Translation.y;
      if not (axZ in OriginAxisMask) then NewTransform.Translation.z:=OriginalTransform.Translation.z;
    end;
    ApplyTransform(NewTransform);
  end;
end;

procedure T3DObjectTransformationWidget.RotationMouseDown(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
var
  RotationAxis: TVector;
begin
  if not BeginTransform(AShift) then Exit;
  if Sender=XRot then
    RotationAxis:=XAxis.Normal
  else if Sender=YRot then
    RotationAxis:=YAxis.Normal
  else if Sender=ZRot then
    RotationAxis:=ZAxis.Normal
  else Exit;
  RotationAxis.Normalize;
  OriginalTransform:=Transform;
  MotionPlane.FromPointAndNormal(Transform.Translation, RotationAxis);
  ActionOffset:=APoint;
  Rotating:=True;
  WidgetManager.ActiveWidget:=T3DWidget(Sender);
end;

procedure T3DObjectTransformationWidget.RotationMouseUp(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  Rotating:=False;
  WidgetManager.ActiveWidget:=nil;
  EndTransform(AShift);
end;

procedure T3DObjectTransformationWidget.RotationMouseMotion(Sender: TObject;
  const ARay: TRay; const APoint: TVector);
var
  NewTransform: TTransform;
  A: Double;
  Mtx, Matrix: TMatrix;
begin
  if Rotating then begin
    A:=AngleBetweenVectorsOnPlane(
      APoint.Subbed(OriginalTransform.Translation),
      ActionOffset.Subbed(OriginalTransform.Translation), WidgetManager.HitPlane.n);
    if WidgetManager.HitPlane.n.Dot(MotionPlane.n) > 0 then A:=-A;
    NewTransform:=OriginalTransform;
    NewTransform.Translation:=Vector(0, 0, 0);
    Matrix:=NewTransform.ToMatrix;
    Mtx.Rotation(MotionPlane.n.x, MotionPlane.n.y, MotionPlane.n.z, A);
    Matrix.SwapMultiply(Mtx);
    NewTransform.FromMatrix(Matrix);
    NewTransform.Translation:=OriginalTransform.Translation;
    NewTransform.Scale:=OriginalTransform.Scale;
    ApplyTransform(NewTransform);
  end;
end;

procedure T3DObjectTransformationWidget.ScaleMouseDown(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  if not BeginTransform(AShift) then Exit;
  OriginalTransform:=Transform;
  if Sender=XScale then
    MovingAxis:=XAxis
  else if Sender=YScale then
    MovingAxis:=YAxis
  else if Sender=ZScale then
    MovingAxis:=ZAxis
  else Exit;
  if ForceUniformScale then
    Uniform:=True
  else begin
    if ssCtrl in AShift then
      if ssAlt in AShift then
        Uniform:=True
      else
        Dual:=True;
  end;
  DistanceBetweenSegments(
    ARay.o, ARay.o.Added(ARay.d.Scaled(163840)),
    Transform.Translation.Subbed(MovingAxis.Normal.Scaled(163840)),
      Transform.Translation.Added(MovingAxis.Normal.Scaled(163840)),
    @ActionOffset, nil);
  ActionOffset:=ProjectVectorOnVector(ActionOffset, MovingAxis.Normal);
  WidgetManager.ActiveWidget:=T3DWidget(Sender);
  Scaling:=True;
end;

procedure T3DObjectTransformationWidget.ScaleMouseUp(Sender: TObject;
  const ARay: TRay; const APoint: TVector; AButton: TMouseButton;
  AShift: TShiftState);
begin
  MovingAxis:=nil;
  Scaling:=False;
  Dual:=False;
  Uniform:=False;
  WidgetManager.ActiveWidget:=nil;
  EndTransform(AShift);
end;

procedure T3DObjectTransformationWidget.ScaleMouseMotion(Sender: TObject;
  const ARay: TRay; const APoint: TVector);
var
  IP: TVector;
  Scale: Double;
  NewTransform: TTransform;
begin
  if Assigned(MovingAxis) and Scaling then begin
    DistanceBetweenSegments(
      ARay.o, ARay.o.Added(ARay.d.Scaled(163840)),
      Transform.Translation.Subbed(MovingAxis.Normal.Scaled(163840)),
        Transform.Translation.Added(MovingAxis.Normal.Scaled(163840)),
      @IP, nil);
    IP:=ProjectVectorOnVector(IP, MovingAxis.Normal);
    Scale:=Distance(OriginalTransform.Translation, IP)/Distance(OriginalTransform.Translation, ActionOffset);
    NewTransform:=OriginalTransform;
    if Uniform then
      NewTransform.Scale.Scale(Scale)
    else if Dual then begin
      if Sender=XScale then begin
        NewTransform.Scale.y *= Scale;
        NewTransform.Scale.z *= Scale;
      end else if Sender=YScale then begin
        NewTransform.Scale.x *= Scale;
        NewTransform.Scale.z *= Scale;
      end else if Sender=ZScale then begin
        NewTransform.Scale.x *= Scale;
        NewTransform.Scale.y *= Scale;
      end else Exit;
    end else begin
      if Sender=XScale then
        NewTransform.Scale.x *= Scale
      else if Sender=YScale then
        NewTransform.Scale.y *= Scale
      else if Sender=ZScale then
        NewTransform.Scale.z *= Scale
      else Exit;
    end;
    ApplyTransform(NewTransform);
  end;
end;

procedure T3DObjectTransformationWidget.SetCoordinateSystem(
  AValue: T3DObjectTransformationWidgetCoordinateSystem);
begin
  if FCoordinateSystem=AValue then Exit;
  FCoordinateSystem:=AValue;
  UpdateFromTransform;
end;

procedure T3DObjectTransformationWidget.SetFlipTowardsCamera(AValue: Boolean);
begin
  if FFlipTowardsCamera=AValue then Exit;
  FFlipTowardsCamera:=AValue;
  UpdateAxisFlip;
end;

procedure T3DObjectTransformationWidget.SetRotationAxes(AValue: TAxes);
begin
  if FRotationAxes=AValue then Exit;
  FRotationAxes:=AValue;
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.SetScaleAxes(AValue: TAxes);
begin
  if FScaleAxes=AValue then Exit;
  FScaleAxes:=AValue;
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.SetShowOrigin(AValue: Boolean);
begin
  if FShowOrigin=AValue then Exit;
  FShowOrigin:=AValue;
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.SetTranslationAxes(AValue: TAxes);
begin
  if FTranslationAxes=AValue then Exit;
  FTranslationAxes:=AValue;
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.UpdateFromTransform;
var
  Matrix, M2, M3: TMatrix;
  FlipX, FlipY, FlipZ: Double;
begin
  case CoordinateSystem of
    tcsLocal: begin
      Matrix:=Transform.ToMatrix;
      Matrix.NormalizeAxes;
    end;
    tcsGlobal: begin
      Matrix.Identity;
      Matrix.SetTranslation(Transform.Translation);
    end;
    tcsCustom: begin
      Matrix.SetAxes(CustomXAxis, CustomYAxis, CustomZAxis, Transform.Translation);
    end;
  end;
  if axX in FlippedAxes then FlipX:=-1 else FlipX:=1;
  if axY in FlippedAxes then FlipY:=-1 else FlipY:=1;
  if axZ in FlippedAxes then FlipZ:=-1 else FlipZ:=1;

  XAxis.Normal:=Matrix.TransformedNormal(Vector(FlipX, 0, 0));
  YAxis.Normal:=Matrix.TransformedNormal(Vector(0, FlipY, 0));
  ZAxis.Normal:=Matrix.TransformedNormal(Vector(0, 0, FlipZ));
  XAxis.Position:=Matrix.Translation;
  YAxis.Position:=Matrix.Translation;
  ZAxis.Position:=Matrix.Translation;
  Origin.Position:=Matrix.Translation;
  XYPlane.Position:=Matrix.Translation;
  XZPlane.Position:=Matrix.Translation;
  ZYPlane.Position:=Matrix.Translation;
  XRot.Position:=Matrix.Translation;
  YRot.Position:=Matrix.Translation;
  ZRot.Position:=Matrix.Translation;
  XScale.Position:=Matrix.Translation;
  YScale.Position:=Matrix.Translation;
  ZScale.Position:=Matrix.Translation;
  // XYPlane
  XYPlane.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(16, 16, 1);
  M2.Multiply(M3);
  M3.Translation(0.75*FlipX, 0.75*FlipY, 0);
  M2.Multiply(M3);
  XYPlane.AddTriangles(CubeVertices, M2);
  // XZPlane
  XZPlane.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(16, 1, 16);
  M2.Multiply(M3);
  M3.Translation(0.75*FlipX, 0, 0.75*FlipZ);
  M2.Multiply(M3);
  XZPlane.AddTriangles(CubeVertices, M2);
  // ZYPlane
  ZYPlane.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(1, 16, 16);
  M2.Multiply(M3);
  M3.Translation(0, 0.75*FlipY, 0.75*FlipZ);
  M2.Multiply(M3);
  ZYPlane.AddTriangles(CubeVertices, M2);
  // XRot
  XRot.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(40);
  M2.Multiply(M3);
  M3.Rotation(0, FlipZ, 0, -PI/2);
  M2.Multiply(M3);
  if axY in FlippedAxes then begin
    M3.Scaling(1, -1, 1);
    M2.Multiply(M3);
  end;
  M3.Translation(0.25, 0.25, 0);
  M2.Multiply(M3);
  XRot.AddTriangles(ArcCornerVertices, M2);
  // YRot
  YRot.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(40);
  M2.Multiply(M3);
  M3.Rotation(FlipZ, 0, 0, PI/2);
  M2.Multiply(M3);
  if axX in FlippedAxes then begin
    M3.Rotation(0, 1, 0, PI);
    M2.Multiply(M3);
  end;
  M3.Translation(0.25, 0.25, 0);
  M2.Multiply(M3);
  YRot.AddTriangles(ArcCornerVertices, M2);
  // ZRot
  ZRot.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(40);
  M2.Multiply(M3);
  if axX in FlippedAxes then begin
    M3.Rotation(0, 1, 0, PI);
    M2.Multiply(M3);
  end;
  if axY in FlippedAxes then begin
    M3.Scaling(1, -1, 1);
    M2.Multiply(M3);
  end;
  M3.Translation(0.25, 0.25, 0);
  M2.Multiply(M3);
  ZRot.AddTriangles(ArcCornerVertices, M2);
  // XScale
  XScale.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(7);
  M2.Multiply(M3);
  M3.Translation(10*FlipX, 0, 0);
  M2.Multiply(M3);
  XScale.AddTriangles(CubeVertices, M2);
  // YScale
  YScale.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(7);
  M2.Multiply(M3);
  M3.Translation(0, 10*FlipY, 0);
  M2.Multiply(M3);
  YScale.AddTriangles(CubeVertices, M2);
  // ZScale
  ZScale.ClearGeometry;
  M2:=Matrix;
  M2.RemoveTranslation;
  M3.Scaling(7);
  M2.Multiply(M3);
  M3.Translation(0, 0, 10*FlipZ);
  M2.Multiply(M3);
  ZScale.AddTriangles(CubeVertices, M2);
end;

procedure T3DObjectTransformationWidget.UpdateAxisFlip;
var
  FlipAxes: TAxes;
begin
  FlipAxes:=[];
  if not (Assigned(WidgetManager) and Assigned(WidgetManager.Environment)) then Exit;
  if not WidgetManager.Environment.IsActiveEnvironment then Exit;
  if FlipTowardsCamera then begin
    if ((axX in FlippedAxes) and (WidgetManager.Environment.GetForwardDirection.Dot(XAxis.Normal) < 0)) or
       ((not (axX in FlippedAxes)) and (WidgetManager.Environment.GetForwardDirection.Dot(XAxis.Normal) > 0)) then
      FlipAxes += [axX];
    if ((axY in FlippedAxes) and (WidgetManager.Environment.GetForwardDirection.Dot(YAxis.Normal) < 0)) or
       ((not (axY in FlippedAxes)) and (WidgetManager.Environment.GetForwardDirection.Dot(YAxis.Normal) > 0)) then
      FlipAxes += [axY];
    if ((axZ in FlippedAxes) and (WidgetManager.Environment.GetForwardDirection.Dot(ZAxis.Normal) < 0)) or
       ((not (axZ in FlippedAxes)) and (WidgetManager.Environment.GetForwardDirection.Dot(ZAxis.Normal) > 0)) then
      FlipAxes += [axZ];
  end;
  if FlipAxes <> FlippedAxes then begin
    FlippedAxes:=FlipAxes;
    UpdateFromTransform;
  end;
end;

procedure T3DObjectTransformationWidget.UpdateWidgets;

  procedure AddRemove(Widget: T3DWidget; AddIt: Boolean);
  begin
    if AddIt then begin
      if IndexOf(Widget)=-1 then
        Add(Widget);
    end else begin
      if IndexOf(Widget) <> -1 then
        Remove(Widget);
    end;
  end;

begin
  AddRemove(Origin, ShowOrigin);
  AddRemove(XAxis, (twoTranslation in Operations) and (axX in AxisMask) and (axX in TranslationAxes));
  AddRemove(YAxis, (twoTranslation in Operations) and (axY in AxisMask) and (axY in TranslationAxes));
  AddRemove(ZAxis, (twoTranslation in Operations) and (axZ in AxisMask) and (axZ in TranslationAxes));
  AddRemove(XYPlane, (twoTranslation in Operations) and (axX in AxisMask) and (axY in AxisMask) and (axX in TranslationAxes) and (axY in TranslationAxes));
  AddRemove(XZPlane, (twoTranslation in Operations) and (axX in AxisMask) and (axZ in AxisMask) and (axX in TranslationAxes) and (axZ in TranslationAxes));
  AddRemove(ZYPlane, (twoTranslation in Operations) and (axZ in AxisMask) and (axY in AxisMask) and (axZ in TranslationAxes) and (axY in TranslationAxes));
  AddRemove(XRot, (twoRotation in Operations) and (axX in AxisMask) and (axX in RotationAxes));
  AddRemove(YRot, (twoRotation in Operations) and (axY in AxisMask) and (axY in RotationAxes));
  AddRemove(ZRot, (twoRotation in Operations) and (axZ in AxisMask) and (axZ in RotationAxes));
  AddRemove(XScale, (twoScale in Operations) and (axX in AxisMask) and (axX in ScaleAxes));
  AddRemove(YScale, (twoScale in Operations) and (axY in AxisMask) and (axY in ScaleAxes));
  AddRemove(ZScale, (twoScale in Operations) and (axZ in AxisMask) and (axZ in ScaleAxes));
end;

procedure T3DObjectTransformationWidget.AddedToManager;
begin
  inherited AddedToManager;
  UpdateFromTransform;
end;

procedure T3DObjectTransformationWidget.TransformChanged;
begin
  UpdateFromTransform;
end;

procedure T3DObjectTransformationWidget.OperationsChanged;
begin
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.AxisMaskChanged;
begin
  UpdateWidgets;
end;

procedure T3DObjectTransformationWidget.Render;
begin
  UpdateAxisFlip;
  inherited Render;
end;

{ T3DCompoundWidget }

constructor T3DCompoundWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWidgets:=T3DWidgetList.Create(False);
end;

destructor T3DCompoundWidget.Destroy;
begin
  Clear;
  FreeAndNil(FWidgets);
  inherited Destroy;
end;

function T3DCompoundWidget.RayHit(const ARay: TRay; out IP: TVector): Boolean;
begin
  Result:=False;
end;

function T3DCompoundWidget.GetWidgetCount: Integer;
begin
  Result:=FWidgets.Count;
end;

function T3DCompoundWidget.GetWidgets(AIndex: Integer): T3DWidget;
begin
  Result:=FWidgets[AIndex];
end;

procedure T3DCompoundWidget.AddedToManager;
var
  I: Integer;
begin
  for I:=0 to WidgetCount - 1 do WidgetManager.Add(Widgets[I]);
end;

procedure T3DCompoundWidget.RemovedFromManager;
var
  I: Integer;
begin
  for I:=0 to WidgetCount - 1 do Widgets[I].WidgetManager.Remove(Widgets[I]);
end;

procedure T3DCompoundWidget.Render;
begin
end;

procedure T3DCompoundWidget.Clear;
begin
  while WidgetCount > 0 do Delete(Widgets[WidgetCount - 1]);
end;

procedure T3DCompoundWidget.Add(AWidget: T3DWidget);
begin
  Assert(AWidget.WidgetManager=nil, 'Tried to add a widget that is already part of a 3d widget manager');
  Assert(FWidgets.IndexOf(AWidget)=-1, 'Tried to add a widget that is already part of this compound widget');
  FWidgets.Add(AWidget);
  if Assigned(WidgetManager) then WidgetManager.Add(AWidget);
end;

procedure T3DCompoundWidget.Remove(AWidget: T3DWidget);
begin
  Assert(FWidgets.IndexOf(AWidget) <> -1, 'Tried to remove a widget that is not part of this compound widget');
  if Assigned(WidgetManager) then WidgetManager.Remove(AWidget);
  FWidgets.Remove(AWidget);
end;

procedure T3DCompoundWidget.Delete(AWidget: T3DWidget);
begin
  Remove(AWidget);
  FreeAndNil(AWidget);
end;

function T3DCompoundWidget.IndexOf(AWidget: T3DWidget): Integer;
begin
  Result:=FWidgets.IndexOf(AWidget);
end;

{ T3DDirectionalWidget }

constructor T3DDirectionalWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FNormal:=Vector(0, 1, 0);
end;

procedure T3DDirectionalWidget.Render;
var
  WorldSize: Double;
  Matrix, TmpMtx: TMatrix;
begin
  UpdatePixelScale;
  if IsMouseWidget then WorldSize:=52*PixelScale else WorldSize:=48*PixelScale;
  Matrix.Direction(Normal);
  TmpMtx.Scaling(WorldSize/32, WorldSize/32, WorldSize);
  Matrix.Multiply(TmpMtx);
  glPushMatrix();
  glTranslated(Position.x, Position.y, Position.z);
  DrawOpenGLTrianglesFor(CylinderVertices, Matrix, Color);
  Matrix.Direction(Normal);
  TmpMtx.Scaling(WorldSize/8, WorldSize/8, WorldSize/4);
  Matrix.Multiply(TmpMtx);
  if Style=dwsArrow then begin
    glTranslated(Normal.x*WorldSize, Normal.y*WorldSize, Normal.z*WorldSize);
    DrawOpenGLTrianglesFor(ConeVertices, Matrix, Color);
  end;
  glPopMatrix();
end;

function T3DDirectionalWidget.RayHit(const ARay: TRay; out IP: TVector): Boolean;
begin
  UpdatePixelScale;
  Result:=DistanceBetweenSegments(ARay.o, ARay.o.Added(ARay.d.Scaled(16384)), Position, Position.Added(Normal.Scaled(PixelScale*48)), @IP, nil) <= PixelScale*8;
end;

{ T3DSphericalWidget }

constructor T3DSphericalWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FNormalRadius:=4;
  FHoverRadius:=7;
end;

procedure T3DSphericalWidget.Render;
var
  Matrix: TMatrix;
begin
  UpdatePixelScale;
  if IsMouseWidget then Matrix.Scaling(HoverRadius*PixelScale) else Matrix.Scaling(NormalRadius*PixelScale);
  glPushMatrix();
  glTranslated(Position.x, Position.y, Position.z);
  DrawOpenGLTrianglesFor(SphereVertices, Matrix, Color);
  glPopMatrix();
end;

function T3DSphericalWidget.RayHit(const ARay: TRay; out IP: TVector): Boolean;
begin
  UpdatePixelScale;
  Result:=ARay.SphereHit(Position, HoverRadius*PixelScale, IP);
end;

{ T3DScaledWidget }

procedure T3DScaledWidget.SetPosition(AValue: TVector);
begin
  if FPosition=AValue then Exit;
  FPosition:=AValue;
  UpdatePixelScale;
end;

procedure T3DScaledWidget.UpdatePixelScale;
begin
  if Assigned(WidgetManager) and Assigned(WidgetManager.Environment) then begin
    FPixelScale:=WidgetManager.Environment.GetPixelScale(Position);
    if FPixelScale < 0 then FPixelScale:=0;
  end;
end;

{ T3DWidget }

constructor T3DWidget.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FNormalColor:=ExtColor(0.8, 0.3, 0.1);
  FHoverColor:=ExtColor(1.0, 1.0, 0.6);
end;

destructor T3DWidget.Destroy;
begin
  if Assigned(FWidgetManager) then FWidgetManager.Remove(Self);
  inherited Destroy;
end;

function T3DWidget.IsMouseWidget: Boolean;
begin
  Result:=Assigned(WidgetManager) and (WidgetManager.MouseWidget=Self)
end;

function T3DWidget.Color: TExtColor;
begin
  if IsMouseWidget then Result:=HoverColor else Result:=NormalColor;
end;

procedure T3DWidget.MouseDown(const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
begin
  if Assigned(OnMouseDown) then OnMouseDown(Self, ARay, APoint, AButton, AShift);
end;

procedure T3DWidget.MouseUp(const ARay: TRay; const APoint: TVector; AButton: TMouseButton; AShift: TShiftState);
begin
  if Assigned(OnMouseUp) then OnMouseUp(Self, ARay, APoint, AButton, AShift);
end;

procedure T3DWidget.MouseMotion(const ARay: TRay; const APoint: TVector);
begin
  if Assigned(OnMouseMotion) then OnMouseMotion(Self, ARay, APoint);
end;

procedure T3DWidget.MouseEnter;
begin
  if Assigned(OnMouseEnter) then OnMouseEnter(Self);
end;

procedure T3DWidget.MouseLeave;
begin
  if Assigned(OnMouseLeave) then OnMouseLeave(Self);
end;

procedure T3DWidget.AddedToManager;
begin
end;

procedure T3DWidget.RemovedFromManager;
begin
end;

procedure T3DWidget.Activated;
begin
  if Assigned(OnActivated) then OnActivated(Self);
end;

procedure T3DWidget.Deactivated;
begin
  if Assigned(OnDeactivated) then OnDeactivated(Self);
end;

{ T3DWidgetManager }

function T3DWidgetManager.GetWidgetCount: Integer;
begin
  Result:=FWidgets.Count;
end;

function T3DWidgetManager.GetWidgets(AIndex: Integer): T3DWidget;
begin
  Result:=FWidgets[AIndex];
end;

procedure T3DWidgetManager.SetActiveWidget(AValue: T3DWidget);
begin
  if FActiveWidget=AValue then Exit;
  if Assigned(FActiveWidget) then FActiveWidget.Deactivated;
  FActiveWidget:=AValue;
  if Assigned(ActiveWidget) then begin
    Assert(FWidgets.IndexOf(ActiveWidget) <> -1, 'The new active widget is not part of this manager');
    ActiveWidget.Activated;
  end;
end;

constructor T3DWidgetManager.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWidgets:=T3DWidgetList.Create(False);
end;

destructor T3DWidgetManager.Destroy;
begin
  Clear;
  FreeAndNil(FWidgets);
  inherited Destroy;
end;

function T3DWidgetManager.FindWidgetByRay(const ARay: TRay; out IP: TVector): T3DWidget;
var
  Distance, ClosestDistance: Double;
  WidIP: TVector;
  I: Integer;
begin
  if Assigned(ActiveWidget) and ActiveWidget.RayHit(ARay, IP) then
    Exit(ActiveWidget);
  Result:=nil;
  ClosestDistance:=MaxInt;
  for I:=0 to WidgetCount - 1 do begin
    if Widgets[I].RayHit(ARay, WidIP) then begin
      Distance:=DistanceSq(ARay.o, WidIP);
      if (not Assigned(Result)) or (ClosestDistance > Distance) then begin
        ClosestDistance:=Distance;
        IP:=WidIP;
        Result:=Widgets[I];
      end;
    end;
  end;
  if not Assigned(Result) then IP:=ARay.o.Added(ARay.d.Scaled(10000));
end;

procedure T3DWidgetManager.MouseDown(const ARay: TRay; AButton: TMouseButton; AShift: TShiftState);
var
  IP: TVector;
  Widget: T3DWidget;
begin
  if not Assigned(Environment) then Exit;
  Widget:=FindWidgetByRay(ARay, IP);
  if Assigned(Widget) then begin
    FHitPlane.FromPointAndNormal(IP, Environment.GetForwardDirection);
    Widget.MouseDown(ARay, IP, AButton, AShift);
  end;
end;

procedure T3DWidgetManager.MouseUp(const ARay: TRay; AButton: TMouseButton; AShift: TShiftState);
var
  IP: TVector;
begin
  if not Assigned(Environment) then Exit;
  if Assigned(ActiveWidget) then begin
    if ARay.PlaneHit(HitPlane, IP) then begin
      ActiveWidget.MouseUp(ARay, IP, AButton, AShift);
    end;
  end;
end;

procedure T3DWidgetManager.MouseMotion(const ARay: TRay);
var
  IP: TVector;
  Widget: T3DWidget;
begin
  if not Assigned(Environment) then Exit;
  if Assigned(ActiveWidget) then begin
    if (ARay.PlaneHit(HitPlane, IP)) then begin
      ActiveWidget.MouseMotion(ARay, IP);
    end;
  end else begin
    Widget:=FindWidgetByRay(ARay, IP);
    if Widget <> MouseWidget then begin
      if Assigned(MouseWidget) then MouseWidget.MouseLeave;
      FMouseWidget:=Widget;
      if Assigned(MouseWidget) then MouseWidget.MouseEnter;
    end;
    if Assigned(Widget) then Widget.MouseMotion(ARay, IP);
  end;
end;

procedure T3DWidgetManager.Render;
var
  I: Integer;
begin
  if not Assigned(Environment) then Exit;
  for I:=0 to WidgetCount - 1 do Widgets[I].Render;
end;

procedure T3DWidgetManager.Clear;
var
  I: Integer;
begin
  // Remove compound widgets first since they can reference other widgets
  for I:=FWidgets.Count - 1 downto 0 do
    if FWidgets[I] is T3DCompoundWidget then
      Delete(FWidgets[I]);
  while FWidgets.Count > 0 do Delete(FWidgets.Last);
end;

procedure T3DWidgetManager.Add(AWidget: T3DWidget);
begin
  Assert(FWidgets.IndexOf(AWidget)=-1, 'Widget is already part of the manager');
  FWidgets.Add(AWidget);
  AWidget.FWidgetManager:=Self;
  AWidget.AddedToManager;
end;

procedure T3DWidgetManager.Remove(AWidget: T3DWidget);
begin
  Assert(FWidgets.IndexOf(AWidget) <> -1, 'Widget is not part of the manager');
  if AWidget=ActiveWidget then begin
    ActiveWidget.Deactivated;
    FActiveWidget:=nil;
  end;
  if AWidget=MouseWidget then begin
    MouseWidget.MouseLeave;
    FMouseWidget:=nil;
  end;
  FWidgets.Remove(AWidget);
  AWidget.FWidgetManager:=nil;
  AWidget.RemovedFromManager;
end;

procedure T3DWidgetManager.Delete(AWidget: T3DWidget);
begin
  Remove(AWidget);
  FreeAndNil(AWidget);
end;

function T3DWidgetManager.IndexOf(AWidget: T3DWidget): Integer;
begin
  Result:=FWidgets.IndexOf(AWidget);
end;

function T3DWidgetManager.Has(AWidget: T3DWidget): Boolean;
begin
  Result:=FWidgets.IndexOf(AWidget) <> -1;
end;

end.

